#!/go/local/bin/wish8.0
# GUI for the Heat Transfer Program
#
# GUI designed by Christopher White,
# for the LCS&E Summer High School Outreach Program
# Email: white@lcse.umn.edu
# 
# Heat Transfer program written by Wenlong Dai
# Email: wenlong@lcse.umn.edu

#########################################################################
# Change the resolution of the output.  Determines whether to run
# x128 or x256
#########################################################################
proc Change { new_res } {

    global m3
    global resolution

    if {$new_res > 128} {
	$m3 delete 0 1
	$m3 add command -label "   128x128" -command {Change 128}
	$m3 add command -label "*  256x256"
	set resolution 256
    } else { # Must be 128
	$m3 delete 0 1
	$m3 add command -label "*  128x128"
	$m3 add command -label "   256x256" -command {Change 256}
	set resolution 128
    }
}
#########################################################################
# Restart the program, so we can use "file|new"
#########################################################################
proc ReStart { } {
    global firsttime
    global top_temp
    global left_temp
    global right_temp
    global bottom_temp

    set firsttime 1
    set top_temp 0
    set bottom_temp 0
    set right_temp 0
    set left_temp 0

    SetScaleColor .scale#1 blue red -50 250 $left_temp
    SetScaleColor .scale#2 blue red -50 250 $top_temp
    SetScaleColor .scale#3 blue red -50 250 $right_temp
    SetScaleColor .scale#4 blue red -50 250 $bottom_temp

    .canvas#1 delete all
    catch {file delete ~/.spam}
    wm withdraw .
    destroy .mat
    PickMaterial .mat
}

#########################################################################
# Open and play a saved movie
#########################################################################
proc OpenMovie { } {
    set types {
	{{Heat Movie}          {.hmov}    }
    }
    
    # Get the filename of the .hmov (Heat Movie)
    set filename [tk_getOpenFile -filetypes $types]
    
    if [file exists $filename] {
	toplevel .waiting
	wm transient .waiting .
	
	wm geometry .waiting +300+300
	message .waiting.t -relief sunken -font 8x16 \
		-width 350 -text "Opening Archive - Please Wait"
	pack .waiting.t
	update idletask
	
	# Rename it with the .gz extention so gzip will decompress
	file rename $filename $filename.gz
	# Decompress the file
	exec gzip -d $filename.gz
	# Untar it
	exec tar xf $filename
	# Remove "waiting" window
	destroy .waiting
	
	# Check for a new colormap and get the script ready
	PrimeXrazscript
	# View the movie
	exec xraz xrazscript 2> /dev/null
	
	# Re-gzip the tar file
	exec gzip $filename
	# Rename it so it's a .gz file with only a .hmov extention
	file rename $filename.gz $filename
	# Delete the TempXXXX files
	Cleanup
    }
}

#########################################################################
# Save the current movie
#########################################################################
proc SaveMovie { } {
   
    set types {
	{{Heat Movie}          {.hmov}    }
    }

    if [file exists Temp0000] {
	set filename [tk_getSaveFile -filetypes $types]

	toplevel .waiting
	wm transient .waiting .
	
	wm geometry .waiting +300+300
	message .waiting.t -relief sunken -font 8x16 \
		-width 350 -text "Creating Archive - Please Wait"
	
	pack .waiting.t
	update idletask

	# Create tar file with the first frame of the movie
	eval exec tar cf $filename.hmov [glob -nocomplain Temp*]
    
        # gzip the tarball to save space
	exec gzip $filename.hmov
	# Rename the file so that it looks like just an .hmov file
	file rename $filename.hmov.gz $filename.hmov
	destroy .waiting
    } else {
	tk_messageBox -icon info -message "You must create a movie first!"
    }

}

#########################################################################
# Open the script and begin creating picture files
#########################################################################
proc OpenScript { } {

    global color
    global result
    global numdumps
    global outline
    global left_temp
    global right_temp
    global top_temp
    global bottom_temp

    set types {
	{{Heat Scripts}        {.script}        }
    }

    set filename [tk_getOpenFile -filetypes $types]

    if {$filename != ""} {

	# Clean up old script
	file delete ~/.spam
	
	set fileID [open $filename {RDONLY}]

	# Clean the canvas
	.canvas#1 delete all

	# This is the blank line in the start of the script
	gets $fileID result
	exec echo $result > ~/.spam
	# This is the number of frames
	gets $fileID numdumps
	exec echo $numdumps >> ~/.spam

	# This is the main material type
	gets $fileID result
	exec echo $result >> ~/.spam
	# Decrement value to get correct color
	set result [expr $result -1]
	switch -exact [expr $result +1] {
	    1 {set material_type gold}
	    2 {set material_type copper}
	    3 {set material_type aluminum}
	    4 {set material_type lead}
	    5 {set material_type iron}
	    6 {set material_type mercury}
	    7 {set material_type air}
	    8 {set material_type water}
	    9 {set material_type alcohol}
	    10 {set material_type oil}
	    11 {set material_type ice}
	    12 {set material_type "wet soil"}
	    13 {set material_type brick}
	    14 {set material_type glass}
	    15 {set material_type "dry soil"}
	    16 {set material_type cork}
	    17 {set material_type oak}
	    18 {set material_type rubber}
	    default {set material_type ERROR}
	}
	color_chooser
	.canvas#1 configure -background $color

	# We check to see if the base color is black (oil)  If so, we draw
	# our box lines in white, otherwise, we draw them in black.
	if {[string compare $color black] == 0} {
	    set outline white
	} else {
	    set outline black
	}   

	# This gets the main temperature
	gets $fileID material_temp
	exec echo $material_temp >> ~/.spam
	.label#1 configure -text "Main Material: $material_type      Main Temperature: $material_temp"

	gets $fileID result
	# We'll loop until we've printed all of the boxes (result=no)
	while {[string compare $result no]} {
	    exec echo $result >> ~/.spam
	    # Material Type
	    gets $fileID result
	    exec echo $result >> ~/.spam
	    # Used to get correct material color, since it adds 1
	    set result [expr $result -1]
	    color_chooser
	    # Read in temperature -- we move it to the new script, that's it
	    gets $fileID result
	    exec echo $result >> ~/.spam

	    # READ IN "IS IT CONSTANT?"
	    gets $fileID const
	    exec echo $const >> ~/.spam

	    # Now let's get the 4 coords for this box
	    gets $fileID x1
	    exec echo $x1 >> ~/.spam
	    gets $fileID y1
	    exec echo $y1 >> ~/.spam
	    gets $fileID x2
	    exec echo $x2 >> ~/.spam
	    gets $fileID y2
	    exec echo $y2 >> ~/.spam

	    # Manipulate the values to fit on our canvas
	    set x1 [expr $x1 / 0.0206]
	    set y1 [expr $y1 / 0.023]
	    set x2 [expr $x2 / 0.0206]
	    set y2 [expr $y2 / 0.023]
	    
	    # Now we draw the box - if constant, outline red, otherwise, not
	    if {[string compare $const no] == 0} {
	    .canvas#1 create rect $x1 $y1 $x2 $y2 -fill $color -outline $outline
	    } else {
		.canvas#1 create rect $x1 $y1 $x2 $y2 -fill $color -outline red
	    }   

	    update idletasks
	    # And read the next line -- is it "yes" or "no"?
	    gets $fileID result
	}

	# We read in the boundary conditions,
	gets $fileID left_temp
	gets $fileID right_temp
	gets $fileID top_temp
	gets $fileID bottom_temp
	# and then change the scale bars
	SetScaleColor .scale#1 blue red -50 250 $left_temp
	SetScaleColor .scale#2 blue red -50 250 $top_temp
	SetScaleColor .scale#3 blue red -50 250 $right_temp
	SetScaleColor .scale#4 blue red -50 250 $bottom_temp
    }
}

#########################################################################
# Save the script
#########################################################################
proc SaveScript { } {
    global right_temp
    global left_temp
    global top_temp
    global bottom_temp
    global x1
    global x2
    global y1
    global y2
    global firsttime

    if {$firsttime == 0} {
	if {$x1 > $x2} {
	    set temp $x2
	    set x2 $x1
	    set x1 $temp
	}
	if {$y1 > $y2} {
	    set temp $y2
	    set y2 $y1
	    set y1 $temp
	}
	CoordCheck
    }
    
    set types { 
	{{Heat Scripts}        {.script}        }
    }
    set filename [tk_getSaveFile -filetypes $types]
    
    if {$filename != ""} {
	file copy ~/.spam $filename.script
	# We have at least one box, and need to flush the coords
	if {$firsttime == 0} {
	    exec echo $x1 >> $filename.script
	    exec echo $y1 >> $filename.script
	    exec echo $x2 >> $filename.script
	    exec echo $y2 >> $filename.script
	}
	exec echo no >> $filename.script
	exec echo $left_temp >> $filename.script
	exec echo $right_temp >> $filename.script
	exec echo $top_temp >> $filename.script
	exec echo $bottom_temp >> $filename.script
    }
}

#########################################################################
# Check if a new colormap exists; if so, use it, otherwise, we'll stick
# with the default one that I made.
#########################################################################
proc PrimeXrazscript { } {

    global cmap
    global cmapname
    global resolution

    # First, let's get a clean copy of the script
    if [file exists xrazscript] {
	file delete xrazscript
    }

    if [file exists .tbar] {
	file delete .tbar
    }

    if {$resolution > 128} {
	file copy xrazscript.256 xrazscript
    } else {
	file copy xrazscript.128 xrazscript
    }

    # If new cmap exists, we'll append that to the end of the script
    # and use that color map
    if {$cmap > 0} {
	exec cmap_convert $cmapname .tbar
	exec echo set colors >> xrazscript
	exec cat $cmapname >> xrazscript
    } else {
	file copy origcmap .tbar
### I added this 4/15/99 AWE
	exec echo set colors >> xrazscript
	exec cat wenlongcmap >> xrazscript
    }

    # Force xraz to start playing once the images are loaded
    exec echo play >> xrazscript
}

#########################################################################
# Display a dialog box; message=the message to display, ysize is the
# y dimension of the box
#########################################################################
proc About { message ysize} {
    toplevel .about
    wm transient .about .

    wm geometry .about 300x$ysize+300+300

    message .about.t -relief sunken -font 8x16 \
	    -width 350 -text $message

    button .about.b1 -text "Ok" -command {destroy .about}

    pack .about.t
    pack .about.b1 -fill y -padx 1m -pady 1m

    update idletask
    grab .about
    tkwait window .about
}

#########################################################################
# Cleanup old TempXXXX files when quiting or starting new run
#########################################################################
proc Cleanup { } {

    if [file exists Temp0000] {
    	eval file delete -force [glob -nocomplain Temp????]
    }
}

#########################################################################
# Let's delete other clutter we may have left laying around
#########################################################################
proc Quitting { } {

    if [file exists .junk] {
	file delete .junk
    }   
    if [file exists .tbar] {
	file delete .tbar
    }   
    if [file exists ~/.spam] {
	file delete ~/.spam
    }   
    if [file exists xrazscript] {
	file delete xrazscript
    }
}

#########################################################################
# Scale color changing function
#########################################################################
proc SetScaleColor {w minColor maxColor minValue maxValue value} {
    set rgb1 [winfo rgb $w $minColor]
    set rgb2 [winfo rgb $w $maxColor]
    foreach {r1 g1 b1} $rgb1 {}
    foreach {r2 g2 b2} $rgb2 {}
    set frac [expr double($value - $minValue) / \
                   double($maxValue - $minValue)]
    foreach col {r g b} {
        set $col [expr int((($frac * ([set ${col}2] - [set ${col}1])) + \
                            [set ${col}1]) / 256)]
    }
    $w configure -troughcolor [format #%02X%02X%02X $r $g $b]
    return
}

#########################################################################
# Procedure for deciding what color each material should be
#########################################################################
proc color_chooser { } {
    global color
    global result

    switch -exact [expr $result + 1] {
	1 {set color gold}
	2 {set color #c06000}
	3 {set color lightgrey}
	4 {set color grey}
	5 {set color darkgrey}
	6 {set color #cacaca}
	7 {set color white}
	8 {set color #2020FF}
	9 {set color beige}
	10 {set color black}
	11 {set color lightblue}
	12 {set color #c0743a}
	13 {set color brown}
	14 {set color cyan}
	15 {set color #3d0e00}
	16 {set color tan}
	17 {set color #734500}
	18 {set color darkgreen}
	default {set color orange}
    }
}

#########################################################################
# Procedure for putting animated dots on the "creating image" window
#########################################################################
proc make_dots { } {
    global dots
    global numdumps
#    append dots *
#    .startwin.label#2 configure -text $dots
#    if {$dots == "************"} {
#	set dots ""
#    }

    set dots [glob -nocomplain Temp*]
    .startwin.label#2 configure -text "Frames Completed: [llength $dots]"

    # If we don't have all of the images yet, keep going
    # Otherwise, change "Stop" to "Done" and remove all references to
    # killing the heat-transfer binary process.
    if {$numdumps > [llength $dots]} {
	after 500 make_dots
    } else {
	.startwin.label#2 configure -text "Frames Completed: DONE"
	.startwin.button#7 \
		configure -command {wm withdraw .startwin
	grab release .startwin
	after cancel make_dots}
	.startwin.button#7 configure -text "Done"
    }
}

#####################################################################
# Create a window for the "Creating Image Files" box
#####################################################################
proc StartWindow {w} {
    toplevel $w
    wm title $w "Creating Image Files..."

    frame .startwin.frame#3
    
    label .startwin.label#6 \
	    -cursor watch
    
    label .startwin.label#1 \
	    -cursor watch \
	    -text {Creating Image Files}
    catch {
	.startwin.label#1 configure \
		-font -*-Helvetica-Bold-R-Normal-*-*-120-*-*-*-*-*-*
    }
    
    label .startwin.label#7 \
	    -cursor watch
    label .startwin.label#5 \
	    -cursor watch
    label .startwin.label#2 \
	    -text "." \
	    -cursor watch    
    label .startwin.label#8 \
	    -cursor watch    
    label .startwin.label#10 \
	    -cursor watch    
    label .startwin.label#9 \
	    -cursor watch    
    label .startwin.label#11 \
	    -cursor watch
    label .startwin.label#13 \
	    -cursor watch
    label .startwin.label#12 \
	    -cursor watch
    label .startwin.label#14 \
	    -cursor watch

    button .startwin.button#6 \
	    -command {PrimeXrazscript
    exec xraz xrazscript 2> /dev/null &} \
	    -text Movie

    button .startwin.button#7 \
	    -command {wm withdraw .startwin
    grab release .startwin
    after cancel make_dots
    # Let's get the process line, and kill it
    set username [exec whoami]
    catch { set killwhat [exec ps -u $username | grep get4d] }
    catch { exec kill -HUP $killwhat}
    StopButton} \
	    -text Stop

    ############################################################
    # Geometry management
    ############################################################
    grid .startwin.frame#3 -in .startwin	-row 3 -column 2 
    grid .startwin.label#6 -in .startwin	-row 1 -column 1  \
	    -sticky nesw
    grid .startwin.label#1 -in .startwin	-row 1 -column 2  \
	    -sticky nesw
    grid .startwin.label#7 -in .startwin	-row 1 -column 3  \
	    -sticky nesw
    grid .startwin.label#5 -in .startwin	-row 2 -column 1  \
	    -sticky nesw
    grid .startwin.label#2 -in .startwin	-row 2 -column 2  \
	    -sticky nesw
    grid .startwin.label#8 -in .startwin	-row 2 -column 3  \
	    -sticky nesw
    grid .startwin.label#10 -in .startwin	-row 3 -column 1  \
	    -sticky nesw
    grid .startwin.label#9 -in .startwin	-row 3 -column 3  \
	    -sticky nesw
    grid .startwin.button#6 -in .startwin.frame#3	-row 1 -column 1  \
	    -sticky w
    grid .startwin.label#11 -in .startwin.frame#3	-row 1 -column 2  \
	    -sticky nesw
    grid .startwin.button#7 -in .startwin.frame#3	-row 1 -column 3  \
	    -sticky e
    grid .startwin.label#13 -in .startwin	-row 4 -column 1  \
	    -sticky nesw
    grid .startwin.label#12 -in .startwin	-row 4 -column 2  \
	    -sticky nesw
    grid .startwin.label#14 -in .startwin	-row 4 -column 3  \
	    -sticky nesw

    # Resize behavior management
    grid rowconfigure .startwin.frame#3 1 -weight 0 -minsize 30
    grid columnconfigure .startwin.frame#3 1 -weight 0 -minsize 38
    grid columnconfigure .startwin.frame#3 2 -weight 0 -minsize 47
    grid columnconfigure .startwin.frame#3 3 -weight 0 -minsize 54
    
    grid rowconfigure $w 1 -weight 0 -minsize 30
    grid rowconfigure $w 2 -weight 0 -minsize 40
    grid rowconfigure $w 3 -weight 0 -minsize 30
    grid rowconfigure $w 4 -weight 0 -minsize 2
    grid columnconfigure $w 1 -weight 0 -minsize 30
    grid columnconfigure $w 2 -weight 0 -minsize 159
    grid columnconfigure $w 3 -weight 0 -minsize 30
}

#####################################################################
# This proc does nothing; Used because wm protocol needs a non-empty
# string, or it reverts to the normal behavior.  This is used to 
# avoid users from closing the material window
#####################################################################
proc DoNothing { } {
# This proc does nothing.  Used because wm protocol needs a non-empty string
}

#####################################################################
# Create a window to select the material type and temperature
#####################################################################
proc PickMaterial {w} {
    global color
    global outline
    global result
    global x1
    global x2
    global y1
    global y2

    toplevel $w
    wm title $w "Material/Temperature"
    wm protocol $w WM_DELETE_WINDOW {DoNothing}

    label $w.label#1 \
	    -text "Main Material"
    
    listbox $w.listbox#1 \
	    -height 6 \
	    -width 0 \
	    -yscrollcommand {.mat.scrollbar#1 set }


    scrollbar $w.scrollbar#1 \
	    -command {.mat.listbox#1 yview} \
	    -orient vertical

    $w.listbox#1 insert end gold copper aluminum lead iron \
	    mercury air water alcohol oil ice "wet soil" brick \
	    glass "dry soil" cork oak rubber

    bind $w.listbox#1 <ButtonRelease-1> {
        set result [.mat.listbox#1 curselection]
    }

    scale $w.scale#1 \
	    -bigincrement 1.0 \
	    -command [list SetScaleColor .mat.scale#1 blue red -50 250] \
	    -highlightthickness 0 \
	    -label {Temp of Material (F)} \
	    -orient h \
	    -resolution 0.1 \
	    -from -50.0 \
	    -to 250.0 \
	    -troughcolor blue \
	    -variable material_temp

    bind $w.scale#1 <Enter> {focus .mat.scale#1}

    # const_temp used as filler here, until checkbutton can be put in
    frame $w.const_temp
    # Placeholder used to separate button from edge of window
    label $w.placeholder

    button $w.button#1 \
	    -command {

	# If this is first time, we'll add dump frequency to the script
	# and add a new checkbox for "maintain const temp"
	if {$firsttime == 1} {
	    exec echo $numdumps >> ~/.spam
	    destroy .mat.const_temp
	    checkbutton .mat.const_temp \
		    -text {Maintain constant temperature} \
		    -variable const
	    grid .mat.const_temp -in .mat   -row 5 -column 2
	}   

	# This puts material type in script
	exec echo [expr $result +1] >> ~/.spam
	# This puts material temperature in script
	exec echo $material_temp >> ~/.spam

	# CONSTANT TEMPERATURE SECTION
	if {$firsttime == 0} {
	    if {$const == 0} {
		exec echo no >> ~/.spam
	    } elseif {$const == 1} {
		exec echo yes >> ~/.spam
	    } else {
		puts stdout ERROR
	    }
	}

	if {$firsttime == 1} {
	    switch -exact [expr $result +1] {
		1 {set material_type gold}
		2 {set material_type copper}
		3 {set material_type aluminum}
		4 {set material_type lead}
		5 {set material_type iron}
		6 {set material_type mercury}
		7 {set material_type air}
		8 {set material_type water}
		9 {set material_type alcohol}
		10 {set material_type oil}
		11 {set material_type ice}
		12 {set material_type "wet soil"}
		13 {set material_type brick}
		14 {set material_type glass}
		15 {set material_type "dry soil"}
		16 {set material_type cork}
		17 {set material_type oak}
		18 {set material_type rubber}
		default {set material_type ERROR}
	    }
	    color_chooser
	    .canvas#1 configure -background $color
	    .label#1 configure -text "Main Material: $material_type      Main Temperature: $material_temp"
	    if {[string compare $color black] == 0} {
		set outline white
	    } else {
		set outline black
	    }   
	    
	} else {	
	    color_chooser
	    set newx1 [expr $x1 / 0.0206]
	    set newx2 [expr $x2 / 0.0206]
	    set newy1 [expr $y1 / 0.023]
	    set newy2 [expr $y2 / 0.023]
	    if {$const == 1} {
		.canvas#1 create rect $newx1 $newy1 $newx2 $newy2 -fill $color -outline red
	    } else {
		.canvas#1 create rect $newx1 $newy1 $newx2 $newy2 -fill $color -outline $outline
	    }	
	}
	.mat.scale#1 configure -troughcolor blue
	.mat.label#1 configure -text Material
	grab release .mat
	set material_temp 0
	wm deiconify .
	grab release .mat
	set const 0
	wm withdraw .mat} \
		-text OK
	            
    ####################################################################
    # Geometry management
    ####################################################################
    grid $w.label#1 -in $w	-row 1 -column 2  \
	    -sticky nesw
    grid $w.listbox#1 -in $w	-row 2 -column 2  \
	    -sticky nesw
    grid $w.scrollbar#1 -in $w	-row 2 -column 3  \
	    -sticky nsw
    grid $w.scale#1 -in $w	-row 3 -column 2  \
	    -sticky ew
    grid $w.button#1 -in $w	-row 6 -column 3 

    grid $w.const_temp -in $w   -row 5 -column 2
    
    grid $w.placeholder -in $w  -row 7 -column 2

    ####################################################################
    # Resize behavior management
    ####################################################################
    grid rowconfigure $w 1 -weight 0 -minsize 30
    grid rowconfigure $w 2 -weight 1 -minsize 179
    grid rowconfigure $w 3 -weight 0 -minsize 1
    grid rowconfigure $w 4 -weight 0 -minsize 1
    grid rowconfigure $w 5 -weight 0 -minsize 15
    grid rowconfigure $w 6 -weight 0 -minsize 6
    grid rowconfigure $w 7 -weight 1 -minsize 5
    grid columnconfigure $w 1 -weight 0 -minsize 30
    grid columnconfigure $w 2 -weight 1 -minsize 176
    grid columnconfigure $w 3 -weight 0 -minsize 18
    grid columnconfigure $w 4 -weight 0 -minsize 6
    
}

##################################################################
# Check for coordinates that leave the screen
##################################################################
proc CoordCheck { } {

    global x1
    global x2
    global y1
    global y2

    if {$x1 < 0} {
	set x1 0
    } elseif {$x1 > 10} {
	set x1 10
    }   
    if {$x2 < 0} {
	set x2 0
    } elseif {$x2 > 10} {
	set x2 10
    }
    if {$y1 < 0} {
	set y1 0
    } elseif {$y1 > 10} {
	set y1 10
    }
    if {$y2 < 0} {
	set y2 0
    } elseif {$y2 > 10} {
	set y2 10
    }   
}

##################################################################
# Get coordinates for first corner of box
##################################################################
proc BoxBegin { w x y } {
    global box
    global color
    global x1
    global y1
    global x2
    global y2

    global firsttime

    # Let's bind button-up - when the box is done, it grabs the
    # material window, displays it, and unbinds button-up
    bind .canvas#1 <ButtonRelease-1> {grab .mat
      bind .canvas#1 <ButtonRelease-1> { }
      wm deiconify .mat}

    # Firsttime is set to 1 if it is the first box, 0 otherwise
    if {$firsttime == 0} {

	if {$x1 > $x2} {
	    set temp $x2
	    set x2 $x1
	    set x1 $temp
	}
	if {$y1 > $y2} {
	    set temp $y2
	    set y2 $y1
	    set y1 $temp
	}

	CoordCheck

	######################################################################
	# Put the coordinates into the input file
	######################################################################
	exec echo $x1 >> ~/.spam
	exec echo $y1 >> ~/.spam
	exec echo $x2 >> ~/.spam
	exec echo $y2 >> ~/.spam
    }

    if {$firsttime == 1} {
	set firsttime 0
    }
    
    exec echo "y" >> ~/.spam
    set box(anchor) [list $x $y]
    catch {unset box(last)}
    set x1 [expr $x * 0.0206]
    set y1 [expr $y * 0.023]
    set x2 $x1
    set y2 $y1

}

##################################################################
# Drag'n'draw the box, keeping track of last coordinate
##################################################################
proc BoxDrag { w x y } {
    global box
    global x2
    global y2
    global outline
    
    catch {$w delete $box(last)}
    set box(last) [eval {$w create rect} $box(anchor) \
	    {$x $y -tag box} -outline $outline]
    set x2 [expr $x * 0.0206]
    set y2 [expr $y * 0.023]
}

##################################################################
# Code to start the heat transfer program
##################################################################
proc StartButton { } {
    global resolution
    global firsttime
    global left_temp
    global right_temp
    global top_temp
    global bottom_temp
    global x1
    global x2
    global y1
    global y2

    # If we have at least 1 subdomain
    if {$firsttime == 0} {
	if {$x1 > $x2} {
	    set temp $x2
	    set x2 $x1
	    set x1 $temp
	}
	if {$y1 > $y2} {
	    set temp $y2
	    set y2 $y1
	    set y1 $temp
	}
	
	CoordCheck
	######################################################################
	# Put the coordinates into the input file
	######################################################################
	exec echo $x1 >> ~/.spam
	exec echo $y1 >> ~/.spam
	exec echo $x2 >> ~/.spam
	exec echo $y2 >> ~/.spam
    }

    # We're making a separate file to add the "no" and the border temperatures
    # This allows the user to click "stop," modify the screen, then redo the
    # simulation
    file copy ~/.spam ~/.spam2
    exec echo no >> ~/.spam2
    exec echo $left_temp >> ~/.spam2
    exec echo $right_temp >> ~/.spam2
    exec echo $top_temp >> ~/.spam2
    exec echo $bottom_temp >> ~/.spam2

    # If res is 256, then run x256, else run x128
    if {$resolution > 128} {
	exec x256 < ~/.spam2 > /dev/null &
    } else {
	exec x128 < ~/.spam2 > /dev/null &
    }
    make_dots
    wm deiconify .startwin
    grab .startwin
}

##################################################################
# Stop the heat-transfer program
##################################################################
proc StopButton { } {
    global resolution

    file delete ~/.spam2
    # Get username of current user
    set username [exec whoami]
    # Let's get the process line, and kill it
    if {$resolution > 128} {
	catch { set killwhat [exec ps -u $username | grep x256] }
    } else {
	catch { set killwhat [exec ps -u $username | grep x128] }
    }
    catch { exec kill -HUP $killwhat}
}

##################################################################
# Make splash-screen
##################################################################
proc splash_screen { } {
    wm withdraw .
    toplevel .splash -borderwidth 4 -relief raised
    wm overrideredirect .splash 1

    after idle {
	update idletasks
	set xmax [winfo screenwidth .splash]
	set ymax [winfo screenheight .splash]
	set x0 [expr ($xmax-[winfo reqwidth .splash])/2]
	set y0 [expr ($ymax-[winfo reqheight .splash])/2]
	wm geometry .splash "+$x0+$y0"
    }

    label .splash.info \
	    -text "http://www.lcse.umn.edu/specs"
    pack .splash.info -side bottom -fill x
    catch {.splash.info configure \
	    -font -*-helvetica-medium-r-normal--*-100-*}
    
    set imh [image create photo -file lcse.gif]

    label .splash.icon -image $imh
    pack .splash.icon -side left -padx 8 -pady 8
    label .splash.title -text "Heat Transfer Program\nBy Chris White and\nWenlong Dai"
    pack .splash.title -fill x -padx 8 -pady 8
    catch {.splash.title configure \
	    -font -*-helvetica-bold-o-normal--*-140-*}
    label .splash.status -text "Initializing Main Memory"
    pack .splash.status -fill x -pady 8
    catch {.splash.status configure \
	    -font -*-helvetica-medium-r-normal--*-120-*}
    update
    exec sleep 2
   .splash.status configure -text "Initializing Hard Drive"
    update
    exec sleep 2
    .splash.status configure -text "Initializing Video Drivers"
    update
    exec sleep 2
    update

    destroy .splash
#    wm deiconify .
}


##################################################################
# Create interface
##################################################################
proc spectest_ui {root args} {

    global firsttime
    global result
    global cmap
    global cmapname
    global m3
    global resolution

    # This makes a pretty splash-screen for visual purposes only
    splash_screen

    ##############################################################
    # Flag to determine if this is the first box we've made
    ##############################################################
    set firsttime 1
    set resolution 128

    exec echo " " > ~/.spam

    # this treats "." as a special case
    
    if {$root == "."} {
	set base ""
    } else {
	set base $root
    }

    scale $base.scale#1 \
	    -bigincrement 1.0 \
	    -command [list SetScaleColor .scale#1 blue red -50 250] \
	    -highlightthickness 0 \
	    -orient v \
	    -from 250.0 \
	    -to -50.0 \
	    -resolution 0.1 \
	    -troughcolor black \
	    -variable left_temp

    bind $base.scale#1 <Enter> {focus .scale#1}

    scale $base.scale#2 \
	    -bigincrement 1.0 \
	    -command [list SetScaleColor .scale#2 blue red -50 250] \
	    -label {Temperature at Edges (F)} \
	    -highlightthickness 0 \
	    -orient h \
	    -from -50.0 \
	    -to 250.0 \
	    -resolution 0.1 \
	    -troughcolor black \
	    -variable top_temp

    bind $base.scale#2 <Enter> {focus .scale#2}

    scale $base.scale#3 \
	    -bigincrement 1.0 \
	    -command [list SetScaleColor .scale#3 blue red -50 250] \
	    -highlightthickness 0 \
	    -orient v \
	    -from 250.0 \
	    -to -50.0 \
	    -resolution 0.1 \
	    -troughcolor black \
	    -variable right_temp

    bind $base.scale#3 <Enter> {focus .scale#3}
    
    scale $base.scale#4 \
	    -bigincrement 1.0 \
	    -command [list SetScaleColor .scale#4 blue red -50 250] \
	    -highlightthickness 0 \
	    -orient h \
	    -from -50.0 \
	    -to 250.0 \
	    -resolution 0.1 \
	    -troughcolor black \
	    -variable bottom_temp

    bind $base.scale#4 <Enter> {focus .scale#4}
    
    canvas $base.canvas#1 \
	    -background white \
	    -cursor pencil \
	    -height 0 \
	    -width 0

    label $base.label#1 \
            -text "Main Material:      Main Temperature:"

    bind $base.canvas#1 <Triple-3> {puts stdout "GUI Author: Christopher White"
    puts stdout "Email: white@lcse.umn.edu"}
    bind $base.canvas#1 <Button-1> {BoxBegin %W %x %y}
    bind $base.canvas#1 <B1-Motion> {BoxDrag %W %x %y}

    button $base.button#1 \
	    -command {Cleanup
    Quitting
    exit} \
	    -text {Quit}

    button $base.button#2 \
	    -text {Start} \
	    -command {Cleanup
    # Create a color bar
    exec creatCbar -f .junk -s 20 512
    exec cmap_convert $cmapname .tbar
    exec make-tbar &
    StartButton}

    menubutton $base.menu#1 \
	    -text File \
	    -menu $base.menu#1.menu

    set m [menu .menu#1.menu -tearoff 0]
    $m add command -label "New" -command {ReStart}
    $m add separator
    $m add command -label "Load Setup" -command {OpenScript}

    $m add command -label "Save Current Setup As..." -command {SaveScript}
    $m add separator
    $m add command -label "Open Movie" -command {OpenMovie}
    $m add command -label "Save Movie As..." -command {SaveMovie}
    $m add separator
    $m add command -label "Quit" -command {Cleanup
    Quitting
    exit}

    menubutton $base.menu#2 \
	    -text Help \
	    -menu $base.menu#2.menu2

    set m2 [menu .menu#2.menu2 -tearoff 0]
    $m2 add command -label About -command {About "Heat Transfer Program\nVersion 0.9\n\nProgram written by Wenlong Dai\nEmail: wenlong@lcse.umn.edu\n\nGUI Written by: Christopher White\nEmail: white@lcse.umn.edu\n\nCopyright 1998, All Rights Reserved" 215}

    menubutton $base.menu#3 \
	    -text "Resolution" \
	    -menu $base.menu#3.menu3

    set m3 [menu .menu#3.menu3 -tearoff 0]
    $m3 add command -label "*  128x128"
    $m3 add command -label "   256x256" -command {Change 256}


    ##################################################################
    # Geometry management
    ##################################################################
    grid $base.menu#1 -in $root         -row 1 -column 1
    grid $base.menu#2 -in $root         -row 1 -column 6
    grid $base.menu#3 -in $root         -row 1 -column 2
    grid $base.scale#2 -in $root	-row 2 -column 3  \
	    -sticky ew
    grid $base.scale#1 -in $root	-row 3 -column 2  \
	    -sticky ns
    grid $base.canvas#1 -in $root	-row 3 -column 3  \
	    -sticky nesw
    grid $base.scale#3 -in $root	-row 3 -column 4  \
	    -sticky ns
    grid $base.scale#4 -in $root	-row 4 -column 3  \
	    -sticky ew
    grid $base.button#1 -in $root	-row 6 -column 2 
    grid $base.button#2 -in $root	-row 6 -column 4 

    grid $base.label#1 -in $root        -row 6 -column 3
    
    # Resize behavior management
    
    grid rowconfigure $root 1 -weight 0 -minsize 30
    grid rowconfigure $root 2 -weight 0 -minsize 30
    grid rowconfigure $root 3 -weight 1 -minsize 437
    grid rowconfigure $root 4 -weight 0 -minsize 30
    grid rowconfigure $root 5 -weight 0 -minsize 30
    grid rowconfigure $root 6 -weight 0 -minsize 30
    grid rowconfigure $root 7 -weight 0 -minsize 30
    grid columnconfigure $root 1 -weight 0 -minsize 30
    grid columnconfigure $root 2 -weight 0 -minsize 30
    grid columnconfigure $root 3 -weight 1 -minsize 486
    grid columnconfigure $root 4 -weight 0 -minsize 30
    grid columnconfigure $root 5 -weight 0 -minsize 30
    grid columnconfigure $root 6 -weight 0 -minsize 30

    # additional interface code
    set result 0
    PickMaterial .mat
    StartWindow .startwin
    wm withdraw .startwin
    wm withdraw .
    grab .mat
    # end additional interface code
    
}

# MAIN -- this is executed first

# Default is no new colormaps
set cmap 0
set cmapname origcmap

# Let's check command-line arguments
# They must be either:
# heat-transfer
# heat-transfer h
# heat-transfer #
# heat-transfer cmap filename
# heat-transfer cmap filename #

# If heat2d does not exist, we'll quit the program
if {[file exists x128] == 0} {
    puts stdout ""
    puts stdout "Could not find 'x128' binary!"
    puts stdout "Please recompile and place binary in this directory."
    puts stdout ""
    puts stdout "Consult the documentation for more information."
    puts stdout ""
    exit
}

if {[file exists x256] == 0} {
    puts stdout ""
    puts stdout "Could not find 'x256' binary!"
    puts stdout "Please recompile and place binary in this directory."
    puts stdout ""
    puts stdout "Consult the documentation for more information."
    puts stdout ""
    exit
}

# Cleanup in case something backfired earlier
if {[file exists ~/.spam2] == 1} {
    file delete ~/.spam2
}


if {[string compare [lindex $argv 0] "h"] == 0} {
    puts stdout ""
    puts stdout "heat-transfer.tcl version 0.8"
    puts stdout ""
    puts stdout "Usage: heat-transfer.tcl \[cmap file\] \[#\]"
    puts stdout ""
    exit
} elseif {[string compare [lindex $argv 0] "cmap"] == 0} {
    # Does the specified file exist?
    if [file exists [lindex $argv 1]] {
	puts stdout ""
	puts stdout "Using '[lindex $argv 1]' as default colormap."
	puts stdout ""
	# Yes, we have a custom cmap
	set cmap 1
	set cmapname [lindex $argv 1]

	# Now check if we have a 3rd argument (# of frame dumps)
	if {[string compare [lindex $argv 2] ""] == 0} {
	    # We have no 3rd arg, so numdumps defaults to 500
	    set numdumps 500
	} elseif {[regexp {^[0-9]*$} [lindex $argv 2]] == 1} {
	    # We do have a 3rd arg, this is the number of dumps
	    set numdumps [lindex $argv 2]
	    puts stdout "Maximum of $numdumps images."
	    puts stdout ""
	} else {
	    puts stdout "Illegal dump frequency '[lindex $argv 2]'.  Must be a number."
	    puts stdout ""
	    exit
	}   
    } else {
	puts stdout ""
	puts stdout "The colormap '[lindex $argv 1]' does not exist."
	puts stdout ""
	exit
    }
} elseif {[string compare [lindex $argv 0] ""] == 0} {
    # No command-line arguments, let's set numdumps to default value.
    set numdumps 500
} elseif {[regexp {^[0-9]*$} [lindex $argv 0]] == 1} {
    set numdumps [lindex $argv 0]
    puts stdout ""
    puts stdout "Maximum of $numdumps images."
    puts stdout ""
} else {
    puts stdout ""
    puts stdout "Illegal dump frequency '[lindex $argv 0]'.  Must be a number."
    puts stdout ""
    exit
}

wm title . "Heat Transfer program by Wenlong Dai and Christopher White"
# This stops the user from being able to close the window via window manager
wm protocol . WM_DELETE_WINDOW {DoNothing}
spectest_ui .
