////////////////////////////////////////////////////////////////////////////////
//
// Author:   David H. Porter
//           Laboratory for Computational Science & Engineering
//           University of Minnesota
//
////////////////////////////////////////////////////////////////////////////////
/***************************************************************************************
Program:  HVR_SERVER  --  Hierarchical Volume Rendering Component Object Moduel
Copyright (C) 2002  David H. Porter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************************/

#include "stdafx.h"

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#define MAX(a,b) ((a)>(b)?(a):(b))
#define MIN(a,b) ((a)<(b)?(a):(b))


int readCmap( char *clrname, char *alpname, float frgba[256][4] )
{
  FILE *fpa, *fpc;
  int i0, i1, i2, i3, j0, j1, j2, j3;
  unsigned int red0, grn0, blu0, alp0,
               red1, grn1, blu1, alp1,
               red2, grn2, blu2, alp2,
               red3, grn3, blu3, alp3;
  unsigned int rgba[256], ishift;

   /********************************************************************
    *               read in color map                                  *
    ********************************************************************/
    fpa = fopen(alpname,"r");    if(fpa == NULL) { return(-1); }
    fpc = fopen(clrname,"r");    if(fpc == NULL) { return(-2); }

    ishift = (unsigned int)256;
    for(i0=0; i0<64; i0++) {
      i1 = i0 + 64;   i2 = i1 + 64;   i3 = i2 + 64;

      fscanf(fpa, "%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d",
                    &j0, &red0, &grn0, &blu0,
                    &j1, &red1, &grn1, &blu1,
                    &j2, &red2, &grn2, &blu2,
                    &j3, &red3, &grn3, &blu3);
      alp0 = (unsigned int)(0.3*(float)red0+0.6*(float)grn0+0.1*(float)blu0);
      alp1 = (unsigned int)(0.3*(float)red1+0.6*(float)grn1+0.1*(float)blu1);
      alp2 = (unsigned int)(0.3*(float)red2+0.6*(float)grn2+0.1*(float)blu2);
      alp3 = (unsigned int)(0.3*(float)red3+0.6*(float)grn3+0.1*(float)blu3);

      fscanf(fpc, "%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d",
                    &j0, &red0, &grn0, &blu0,
                    &j1, &red1, &grn1, &blu1,
                    &j2, &red2, &grn2, &blu2,
                    &j3, &red3, &grn3, &blu3);

      rgba[i0] = alp0 + ishift * (blu0 + ishift * (grn0 + ishift * red0));
      rgba[i1] = alp1 + ishift * (blu1 + ishift * (grn1 + ishift * red1));
      rgba[i2] = alp2 + ishift * (blu2 + ishift * (grn2 + ishift * red2));
      rgba[i3] = alp3 + ishift * (blu3 + ishift * (grn3 + ishift * red3));

      frgba[i0][0] = (float)red0 / (float)255.;
      frgba[i0][1] = (float)grn0 / (float)255.;
      frgba[i0][2] = (float)blu0 / (float)255.;
      frgba[i0][3] = (float)alp0 / (float)255.;

      frgba[i1][0] = (float)red1 / (float)255.;
      frgba[i1][1] = (float)grn1 / (float)255.;
      frgba[i1][2] = (float)blu1 / (float)255.;
      frgba[i1][3] = (float)alp1 / (float)255.;

      frgba[i2][0] = (float)red2 / (float)255.;
      frgba[i2][1] = (float)grn2 / (float)255.;
      frgba[i2][2] = (float)blu2 / (float)255.;
      frgba[i2][3] = (float)alp2 / (float)255.;

      frgba[i3][0] = (float)red3 / (float)255.;
      frgba[i3][1] = (float)grn3 / (float)255.;
      frgba[i3][2] = (float)blu3 / (float)255.;
      frgba[i3][3] = (float)alp3 / (float)255.;

    }
    fclose(fpa);
    fclose(fpc);

	return(0);
}

/******************************************************************
 *                     basic vector functions                     *
 ******************************************************************/

float dotproduct( const float *v, const float *w )
{
  int i;
  float d;
  d = 0.;
  for ( i=0; i<3; ++i ) d += v[i]*w[i];
  return d;
}  

void vnormalize( float v[3] )
{
  float vt;
  
  vt = (float)1.0 / (float)sqrt( v[0]*v[0] + v[1]*v[1] + v[2]*v[2] );
  v[0] *= vt; v[1] *= vt;  v[2] *= vt;
}

void vcross(const float *v1, const float *v2, float *cross)
{
  int i;
  float t[4];
    /* Vector cross product.
     */
    t[0] = (v1[1] * v2[2]) - (v1[2] * v2[1]);
    t[1] = (v1[2] * v2[0]) - (v1[0] * v2[2]);
    t[2] = (v1[0] * v2[1]) - (v1[1] * v2[0]);
    for ( i=0; i<4; ++i ) cross[i] = t[i];

}

int match( char *token, char *word, int tokenlength ) {
  int n,i;
  n = strlen(word);
  if ( n>tokenlength ) n= tokenlength;
  i = !strncmp(token,word,n);
  return i;
}

// ----------------------------------------------------------------------------------------
// This routine takes a raw (rgb) image and writes it to disc in bmp format.
// David Porter     June    19, 2001     Adapted from Ted Wetherbee's test bmp test code
// Ted Wetherbee    October 24, 2000     Original version

void write_bmp_file(int width, int height, unsigned char *image, char file[256]) {
	// the .bmp header information
	struct info_header{
   		char id1;						// 0
   		char id2;						// 1
   		unsigned int file_length;		// 2 3 4 5
   		unsigned int reserved1;			// 6 7 8 9
   		unsigned int offset;			// 10 11 12 13
   		unsigned int info_length;		// 14 15 16 17
   		unsigned int width;				// 18 19 20 21
   		unsigned int length;			// 22 23 24 25
   		unsigned short biplanes;		// 26 27
   		unsigned short bitcount;		// 28 29
   		unsigned int compression;		// 30 31 32 33
   		unsigned int bisizeimage;		// 34 35 36 37
   		unsigned int Xpixelspermeter;	// 38 39 40 41
   		unsigned int Ypixelspermeter;	// 42 43 44 45
   		unsigned int colorsused;		// 46 47 48 49
   		unsigned int importantcolors;	// 50 51 52 53
	} header = {'B','M',512*512*3+54,0,54,40,512,512,1,24,0,0,0,0,0,0};
	int i;
	FILE *fp;
	unsigned char red, grn, blu;

   header.file_length = (unsigned int)(54 + 3*width*height);
   header.width  = (unsigned int)width;
   header.length = (unsigned int)height;

   // Open file for writing binary info.
   fp = fopen(file, "wb");

   // write the 54 byte .bmp header
   fwrite(&header.id1, sizeof(char), 1, fp);
   fwrite(&header.id2, sizeof(char), 1, fp);
   fwrite(&header.file_length, sizeof(unsigned int), 6, fp);
   fwrite(&header.biplanes,    sizeof(unsigned short), 2, fp);
   fwrite(&header.compression, sizeof(unsigned int), 6, fp);

   // Convert image from raw (rgb) to bmp (bgr) format
   for(i=0; i<3*width*height; i += 3) {
	   red = image[i  ];
	   grn = image[i+1];
	   blu = image[i+2];
	   image[i  ] = blu;
	   image[i+1] = grn;
	   image[i+2] = red;
   }

   // write bmp formatted image to file
   fwrite(image, sizeof(unsigned char), 3*width*height, fp);

   // close bmp file
   fclose(fp);
}