////////////////////////////////////////////////////////////////////////////////
//
// Authors:  David H. Porter
//           Sarah Anderson
//           Laboratory for Computational Science & Engineering
//           University of Minnesota
//
////////////////////////////////////////////////////////////////////////////////
/***************************************************************************************
Program:  HVR_SERVER  --  Hierarchical Volume Rendering Component Object Moduel
Copyright (C) 2002  Sarah Anderson & David H. Porter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************************/

#include "stdafx.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

// #include "renderparams.h"
#include "proto.h"

struct Renderparams Rp;
struct Renderparams Keyframes[256];
struct SLutNots KeySluts[256];
extern struct Renderparams Movieframes[4096];
extern struct SLutNots MovieSluts[4096];
int nkeyframes   = 0;
int ikeyframe    = 0;
int nmovieframes = 0;
int iwrap, nxfull, nyfull, nzfull;
char BrickDir[256];
int nbobs;                              /* number of bob files        */
char bobdir[3000][80];                  /* directories of bob files   */
char bobfile[3000][80];                 /* bobfile names              */

FILE *fperr;

int KeysToHvrIn(char DirPath[3000][80], char BobList[3000][80],
				struct Renderparams Keys[64], struct SLutNots SlutKeys[64], char *movnam, bool bLoopOverData)
{
  char *cKey0, *cKey1;
  int im, i1, i2, i3;

  static char *seps = " ,;\t\n";
  int i;

  fperr = fopen("KeysToHvrIn.out", "w");
  fprintf(fperr, "At top of KeysToHvrIn: movie name is: %s\n", movnam);
  fclose(fperr);

  nbobs = 0;
  while( match(BobList[nbobs],"end",3) != 1 ) {
	  sprintf(bobfile[nbobs], "%s%s", DirPath[nbobs], BobList[nbobs]);
	  nbobs++;
  }
  nxfull = Keys[0].nx;
  nyfull = Keys[0].ny;
  nzfull = Keys[0].nz;

  fperr = fopen("KeysToHvrIn.out", "a+");
  fprintf(fperr, "got boblist & fulldir: nbobs = %d\n", nbobs);
  for(i=0; i<nbobs; i++) { fprintf(fperr, "%3d : %s\n", i, bobfile[i]); }
  fclose(fperr);

  nkeyframes = 0;
  while(Keys[nkeyframes].nfpk != 0) {
	memcpy(&Keyframes[nkeyframes].opacity, &Keys[nkeyframes].opacity, sizeof(Renderparams));
	memcpy(&KeySluts[nkeyframes].iNCnots, &SlutKeys[nkeyframes].iNCnots, sizeof(SLutNots));
	fperr = fopen("KeysToHvrIn.out", "a+");
	fprintf(fperr, "Nfpk = %d     Nrpk = %f     BobFile = %s\n",
		Keyframes[nkeyframes].nfpk, Keyframes[nkeyframes].nrpk, Keyframes[nkeyframes].hvname);
	fclose(fperr);
    nkeyframes++;
  }
  fperr = fopen("KeysToHvrIn.out", "a+");
  fprintf(fperr, "got keyframes: nkeyframes = %d\n", nkeyframes);
  fclose(fperr);

  cKey0 = (char *)&Keyframes[           0].opacity;
  cKey1 = (char *)&Keyframes[nkeyframes-1].opacity;
  iwrap = 1;
  for(i=0; i<sizeof(Renderparams); i++) {
	  if(*(cKey0+i) != *(cKey1+i)) { iwrap = 0; }
  }
  if(interp_path(bLoopOverData) <= 0) return (-1);

  fperr = fopen("KeysToHvrIn.out", "a+");
  fprintf(fperr, "Finished interp and smoothing: nmovieframes = %d\n", nmovieframes);
  fprintf(fperr, "iwrap = %d\n", iwrap);
  fclose(fperr);

    for(im=0; im<nmovieframes; im++) {
		i1 =  im/100    ;
		i2 = (im/10 )%10;
		i3 = (im    )%10;
		sprintf((char *)Movieframes[im].imagename, "%s%d%d%d",	movnam, i1, i2, i3);
/*
      sprintf(trifile, "TRI%d%d%d.tri", i1, i2, i3);
      fpout = fopen(trifile, "w");
      i = rend_dump_params(&Movieframes[im], fpout, movnam, im);
      fclose(fpout);
*/
    }

	return(nmovieframes);
}

void rend_setdefaults( struct Renderparams *rp )
{
  *rp->cmapname = 0;
  *rp->amapname = 0;
  rp->opacity = 1.0;
  rp->nx = rp->ny = rp->nz = 0;
  *rp->bobname = 0;
  rp->hpos = rp->vpos = 0;
  rp->nsectx = rp->nsecty = 1;
  rp->isectx = rp->isecty = 1;
  rp->width = 1024;
  rp->height = 1024;
/*  rp->image = NULL; */
  *rp->imagename= 0;
  *rp->moviename= 0;
  rp->fovangle = 40.0;
  rp->eyeX = -5.0;
  rp->eyeY = 0.;
  rp->eyeZ = 0.;
  rp->centerX = rp->centerY = rp->centerZ= 0;
  rp->upX = rp->upY = 0.;
  rp->upZ = 1.0;
  rp->viewdefined[0] = rp->viewdefined[1] =  rp->viewdefined[2] =
    rp->viewdefined[2] = 0;
  rp->my_near = (float)3.27;      /* unit cube for any orientation */
  rp->my_far  = (float)5.73;
  rp->softdistance = (float)0.;
  rp->nplanes    = (float)4.0;
  rp->maxplanes  = (float)4.0;
  rp->boxcolor = -1;    /* that is, none */
  rp->ifauxclip[0]= rp->ifauxclip[1]= rp->ifauxclip[2]= rp->ifauxclip[3]=
  rp->ifauxclip[4]= rp->ifauxclip[5]= 0;
  rp->nfpk = 1;
  rp->nrpk = 0.;
  rp->ntpf = 0; 

  rp->icoltolerance = 6;
  rp->angletolerance = 0.;
  rp->vrtolerance = -1.;
}

static int match( char *token, char *word, int tokenlength )
{
  int n,i;
  n = strlen(word);
  if ( n>tokenlength ) n= tokenlength;
  i = !strncmp(token,word,n);
  return i;
}


int rend_load_from_file(struct Renderparams *Rp, FILE *fp)
{
  char line[256];
  char *keyword;
  static char *seps = " ,;\t\n";
  int nc;
  int i;

//  printf("just befor start of rend_load_file fgets loop\n");
  while ( fgets(line,80,fp) != NULL ) {
//	  printf("at start of rend_load_file fgets loop\n");
    if ( strncmp(line, "# NEXT KEY",10) == 0 ) return(1);
    if ( line[0] == '#' ) continue;

    nc = strlen(line);
    keyword = strtok( line,seps );
    nc = strlen(keyword);
    if (!nc) continue;

    if ( match(keyword,"color",nc) ) {
      strncpy( (char *)Rp->cmapname, strtok(NULL,seps),80 );
    }
    else if ( match(keyword,"alpha",nc) ) {
      strncpy( (char *)Rp->amapname, strtok(NULL,seps),80 );
    }
    else if ( match(keyword,"opacity",nc) ) {
      Rp->opacity = (float)atof( strtok(NULL,seps) );
    } 
    else if ( match(keyword,"dimensions",nc) ) {
      Rp->nx = atoi( strtok(NULL,seps) );
      Rp->ny = atoi( strtok(NULL,seps) );
      Rp->nz = atoi( strtok(NULL,seps) );
    }
    else if ( match(keyword,"file",nc) ) {
      strncpy( (char *)Rp->bobname, strtok(NULL,seps),80 );
    }
    else if ( match(keyword,"hvfile",nc) ) {
      strncpy( (char *)Rp->hvname, strtok(NULL,seps),80 );
    }
    else if ( match(keyword,"imagefile",nc) ) {
      strncpy( (char *)Rp->imagename, strtok(NULL,seps),80 );
    }
    else if ( match(keyword,"moviefile",nc) ) {
      strncpy( (char *)Rp->moviename, strtok(NULL,seps),80 );
    }
    else if ( match(keyword,"section",nc) ) {
      Rp->isectx = atoi(strtok(NULL,seps) );
      Rp->isecty = atoi(strtok(NULL,seps) );
      Rp->nsectx = atoi(strtok(NULL,seps) );
      Rp->nsecty = atoi(strtok(NULL,seps) );
    }
    else if ( match(keyword,"imageposition",nc) ) {
      Rp->hpos = atoi(strtok(NULL,seps) );
      Rp->vpos = atoi(strtok(NULL,seps) );
    }
    else if ( match(keyword,"imagesize",nc) ) {
      Rp->width = atoi( strtok(NULL,seps) );
      Rp->height = atoi( strtok(NULL,seps) );
    } 
    else if ( match(keyword,"clipping",nc) ) {
      Rp->my_near =  (float)atof( strtok(NULL,seps) );
      Rp->my_far =  (float)atof( strtok(NULL,seps) );
    } 
    else if ( match(keyword,"softdistance",nc) ) {
      Rp->softdistance =  (float)atof( strtok(NULL,seps) );
    } 
    else if ( match(keyword,"nplanes",nc) ) {
      Rp->nplanes    = (float)atof( strtok(NULL,seps)) ;
      Rp->maxplanes = (float)atof( strtok(NULL,seps)) ;
    }
    else if ( match(keyword,"boxcolor",nc) ) {
      Rp->boxcolor = atoi( strtok(NULL,seps)) ;
    }
    else if ( match(keyword,"fov",nc) ) {
      Rp->fovangle = (float)atof( strtok(NULL,seps) ) ;
    }
    else if ( (match(keyword,"eye",nc)) ||
	(match(keyword, "eye", nc)) ){
      Rp->eyeX = (float)atof( strtok(NULL,seps) );
      Rp->eyeY = (float)atof( strtok(NULL,seps) );
      Rp->eyeZ = (float)atof( strtok(NULL,seps) );
      Rp->viewdefined[0] = 1;
    }
    else if ( match(keyword,"center",nc) ) {
      Rp->centerX = (float)atof( strtok(NULL,seps) );
      Rp->centerY = (float)atof( strtok(NULL,seps) );
      Rp->centerZ = (float)atof( strtok(NULL,seps) );
      Rp->viewdefined[1] = 1;
    }
    else if ( match(keyword,"up",nc) ) {
      Rp->upX = (float)atof( strtok(NULL,seps) );
      Rp->upY = (float)atof( strtok(NULL,seps) );
      Rp->upZ = (float)atof( strtok(NULL,seps) );
      Rp->viewdefined[2] = 1;
    }
    else if ( match(keyword,"eyedirection",nc) ) {
      Rp->eyedirX = (float)atof( strtok(NULL,seps) );
      Rp->eyedirY = (float)atof( strtok(NULL,seps) );
      Rp->eyedirZ = (float)atof( strtok(NULL,seps) );
      Rp->viewdefined[3] = 1;
    }
    else if ( match(keyword,"tolerance",nc) ) {
      Rp->icoltolerance  = atoi( strtok(NULL,seps) );
      Rp->angletolerance = (float)atof( strtok(NULL,seps) );
      Rp->vrtolerance = (float)atof( strtok(NULL,seps) );
    }
    else if ( match(keyword,"ifauxclip",nc) ) {
		i = atoi( strtok(NULL,seps) );
		Rp->ifauxclip[i] = atoi( strtok(NULL,seps) );
    }
    else if ( match(keyword,"auxclip",nc) ) {
		i = atoi( strtok(NULL,seps) );
		Rp->auxclip[i][0] = (float)atof( strtok(NULL,seps) );
		Rp->auxclip[i][1] = (float)atof( strtok(NULL,seps) );
		Rp->auxclip[i][2] = (float)atof( strtok(NULL,seps) );
		Rp->auxclip[i][3] = (float)atof( strtok(NULL,seps) );
	}
    else if ( match(keyword,"frameperkey",nc) ) {
      Rp->nfpk = atoi( strtok(NULL,seps) );
    }
    else if ( match(keyword,"timeperframe",nc) ) {
      Rp->ntpf = atoi( strtok(NULL,seps) );
    }
    else if ( match(keyword,"rotperkey",nc) ) {
      Rp->nrpk = (float)atof( strtok(NULL,seps) );
    } else { 
      fprintf(stderr,"***** '%s' undefined keyword ignored\n",keyword);
    }
  } /* end while */
  return(0); 
}

int rend_dump_params(struct Renderparams *rp, FILE *fp, char *movnam, int im)
{
    int i1, i2, i3;
    int i, resolution;
    int divisor;
    static char *bs = "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb";
    char extension[256]; 

    resolution = 0;

    i1 =  im/100    ;
    i2 = (im/10 )%10;
    i3 = (im    )%10;
    
    divisor = (1 << resolution);
    if (resolution > 0)
        strncpy(extension, bs, resolution);
    extension[resolution] = '\0';
    

    fprintf(fp, "color\t\t%s\n",			rp->cmapname);
    fprintf(fp, "alpha\t\t%s\n",			rp->amapname);
    fprintf(fp, "opacity\t\t%f\n",			rp->opacity);
    fprintf(fp, "dimensions\t%d %d %d\n",	rp->nx, rp->ny, rp->nz);
    fprintf(fp, "hvfile\t\t%s\n",			rp->hvname);
    fprintf(fp, "imagesize\t%d, %d\n",		rp->width, rp->height);
    fprintf(fp, "imageposition\t%d, %d\n",	rp->hpos, rp->vpos);
    fprintf(fp, "section\t\t%d,%d  %d,%d\n",rp->isectx,rp->isecty, 
											rp->nsectx,rp->nsecty);
    fprintf(fp, "imagefile\t%s%d%d%d\n",	movnam, i1, i2, i3);

/*
    fprintf(fp, "moviefile\t%s\n",		rp->moviename);
    fprintf(fp, "movieframe\t%d\n",		rp->movieframe);
*/
    fprintf(fp, "fov\t\t%f\n",				rp->fovangle);
    fprintf(fp, "eye\t\t%f, %f, %f\n",		rp->eyeX, rp->eyeY, rp->eyeZ);
    fprintf(fp, "center\t\t%f, %f, %f\n",	rp->centerX, rp->centerY, rp->centerZ );
    fprintf(fp, "up\t\t%f, %f, %f\n",		rp->upX, rp->upY, rp->upZ );
    fprintf(fp, "clipping\t%f, %f\n",		rp->my_near, rp->my_far);
    fprintf(fp, "nplanes\t\t%f, %f\n",		rp->nplanes, rp->maxplanes);
    fprintf(fp, "softdistance\t%f\n",		rp->softdistance);
    fprintf(fp, "boxcolor\t%d\n",			rp->boxcolor);
    fprintf(fp, "tolerance\t%d %f %f\n", 	rp->icoltolerance,
											rp->angletolerance,
											rp->vrtolerance        );
    if (rp->ifauxclip[0] != 0) {
		for(i = 0; i < 6; i++) {
			fprintf(fp, "ifauxclip %d   %d\n", i, rp->ifauxclip[i]);
		}
		for(i = 0; i < 6; i++) {
            fprintf(fp, "auxclip   %d    %f %f %f %f\n", i,
	        rp->auxclip[i][0],rp->auxclip[i][1],rp->auxclip[i][2],rp->auxclip[i][3]);
		}
	}
    fprintf(fp, "frameperkey\t%d\n",		rp->nfpk);
    fprintf(fp, "timeperframe\t%d\n",		rp->ntpf);
    fprintf(fp, "rotperkey\t%f\n",			rp->nrpk);
    fprintf(fp, "render\n"                              );

    return(0);
}

void gen_eye( float xyz[3], float angle, struct Renderparams *rp )
{
    float eX, eY, eZ, hX, hY, hZ, coX, coY, coZ, csa, sna;
    float nupi, nupX, nupY, nupZ, DecX, DecY, DecZ, dec_DOT_nup;

    DecX = rp->eyeX - rp->centerX;
    DecY = rp->eyeY - rp->centerY;
    DecZ = rp->eyeZ - rp->centerZ;

    nupi = (float)1. / (float)sqrt( rp->upX*rp->upX + rp->upY*rp->upY + rp->upZ*rp->upZ );
    nupX = nupi * rp->upX;
    nupY = nupi * rp->upY;
    nupZ = nupi * rp->upZ;

    dec_DOT_nup = DecX * nupX  +  DecY * nupY  +  DecZ * nupZ;

    coX = rp->centerX + nupX * dec_DOT_nup;
    coY = rp->centerY + nupY * dec_DOT_nup;
    coZ = rp->centerZ + nupZ * dec_DOT_nup;

    eX = rp->eyeX - coX;
    eY = rp->eyeY - coY;
    eZ = rp->eyeZ - coZ;

    hX = rp->upY * eZ - nupZ * eY; /* "nup" is othonormal to "e"          */
    hY = rp->upZ * eX - nupX * eZ; /* hence "h" has the same size as "e"  */
    hZ = rp->upX * eY - nupY * eX; /* and is normal to both "e" and "nup" */

    csa = (float)cos(angle);
    sna = (float)sin(angle);

    xyz[0] = csa * eX  +  sna * hX  +  coX;  /* "xyz" orbits around "co"  */
    xyz[1] = csa * eY  +  sna * hY  +  coY;  /* in a plane normal to "up" */
    xyz[2] = csa * eZ  +  sna * hZ  +  coZ;
}

int interp_path(bool bLoopOverData)
{
    int i, i1, i2, ij, j, k, im, ic, jc, n1, n2, nfpk, nfpkabs, inot;
//    int iusebob0;
    float f0, f1, angle, xyz0[3], xyz1[3], fnrpk, pi2, fLimit;
	float ff0[4096], ff1[4096];
	float CN0im[4096], CN1im[4096], FC0im[4096], FC1im[4096];
	float CentNear0, CentNear1, FarCent0, FarCent1, x, y, z, distance, CurrentCentNear;
	int GetTail(char *str, char *tail);
	char bobtail[256], keytail_1[256], keytail_2[256];

    im = 0;
    for(k=1; k<nkeyframes; k++) {

		// Deal with auxclips
		if(Keyframes[k-1].ifauxclip[0] == 2  &&  Keyframes[k].ifauxclip[0] == 0) {
			fLimit = (float)(1.0);
			for(i=0; i<6; i++) { Keyframes[k].ifauxclip[i] = 2;
								 fLimit = -fLimit;
								 Keyframes[k].auxclip[i][3] = fLimit; }
		}
		if(Keyframes[k-1].ifauxclip[0] == 0  &&  Keyframes[k].ifauxclip[0] == 2) {
			fLimit = (float)(1.0);
			for(i=0; i<6; i++) { Keyframes[k-1].ifauxclip[i] = 2;
								 fLimit = -fLimit;
								 Keyframes[k-1].auxclip[i][3] = fLimit; }
		}

		n1 = GetTail((char *)Keyframes[k-1].hvname, keytail_1);
		n2 = GetTail((char *)Keyframes[k  ].hvname, keytail_2);
		i1 = -1;
		i2 = -2;
		for(i=0; i<nbobs; i++) {
			n2 = GetTail(bobfile[i], bobtail);
			if(strncmp(keytail_1, bobtail, n1) == 0 ) { i1 = i; }
			if(strncmp(keytail_2, bobtail, n2) == 0 ) { i2 = i; }
		}
//		  if(i1 < 0 || i2 < 0) {
// 			fperr = fopen("KeysToHvrIn.out", "a+");
//			fprintf(fperr, "PROBLEM: Keyframe HV files are not in HV list\n");
//			fprintf(fperr, "%s   %s\n",Keyframes[k-1].hvname, Keyframes[k].hvname);
//			fprintf(fperr, "keytail_1 = -->%s<--   keytail_2 = -->%s<--\n",keytail_1, keytail_2);
//			fclose(fperr);
//			return (-1);
//		  }
//		  if(i2 < i1) {
//			fperr = fopen("KeysToHvrIn.out", "a+");
//			fprintf(fperr, "PROBLEM: 2nd key frame comes before 1st in bob_list\n");
//			fprintf(fperr, "This option is not implemented\n");
//			fclose(fperr);
//			return (-1);
//		  }
//		  if(i2 == i1) { iusebob0 = 1; nfpk = Keyframes[k].nfpk; }
//		  else         { iusebob0 = 0; nfpk = i2 - i1;           }
//
//		  fperr = fopen("KeysToHvrIn.out", "a+");
//		  fprintf(fperr, "iusebob0=%d    nfpk=%d\n", iusebob0, nfpk);
//		  fclose(fperr);

	  nfpk = Keyframes[k].nfpk;
	  if(nfpk > 0) { nfpkabs = nfpk; } else { nfpkabs = -nfpk; }

      x = Keyframes[k-1].eyeX - Keyframes[k-1].centerX;
      y = Keyframes[k-1].eyeY - Keyframes[k-1].centerY;
      z = Keyframes[k-1].eyeZ - Keyframes[k-1].centerZ;
      distance = (float)sqrt((double)(x*x + y*y + z*z));
	  CentNear0 = distance - Keyframes[k-1].my_near;
	  FarCent0  = Keyframes[k-1].my_far - distance;

      x = Keyframes[k].eyeX - Keyframes[k].centerX;
      y = Keyframes[k].eyeY - Keyframes[k].centerY;
      z = Keyframes[k].eyeZ - Keyframes[k].centerZ;
      distance = (float)sqrt((double)(x*x + y*y + z*z));
	  CentNear1 = distance - Keyframes[k].my_near;
	  FarCent1  = Keyframes[k].my_far - distance;

      for(j=0; j<nfpkabs; j++) {
        memcpy(&Movieframes[im], &Keyframes[k-1], sizeof(Rp));

		if(bLoopOverData) {
			n1 = strlen((char *)Keyframes[0  ].hvname);
			for(i=0; i<n1; i++) { Movieframes[im].hvname[i] = ' '; }
			n1 = strlen(bobfile[im%nbobs]);
			sprintf((char *)Movieframes[im].hvname, "%s", bobfile[im%nbobs]);
		} else {
			ij = (i1 + int( (float)0.5 + (float)j * (float)(i2-i1) / (float)nfpkabs ));
			n1 = strlen((char *)Keyframes[k  ].hvname);
			for(i=0; i<n1; i++) { Movieframes[im].hvname[i] = ' '; }
			n1 = strlen(bobfile[i1+j]);
			sprintf((char *)Movieframes[im].hvname, "%s", bobfile[ij]);

//			if(iusebob0 == 0) {
//			  n1 = strlen((char *)Keyframes[k  ].hvname);
//			  for(i=0; i<n1; i++) { Movieframes[im].hvname[i] = ' '; }
//			  n1 = strlen(bobfile[i1+j]);
//			  sprintf((char *)Movieframes[im].hvname, "%s", bobfile[i1+j]);
//			} else {
//			  n1 = strlen((char *)Keyframes[k  ].hvname);
//			  for(i=0; i<n1; i++) { Movieframes[im].hvname[i] = ' '; }
//			  n1 = strlen(bobfile[i1]);
//			  sprintf((char *)Movieframes[im].hvname, "%s", bobfile[i1]);
//			}
		}

	    fperr = fopen("KeysToHvrIn.out", "a+");
        fprintf(fperr, "-->%s<--\n",Movieframes[im].hvname);
		fclose(fperr);

        f0 = (float)j / (float)nfpkabs;
        f1 = (float)1. - f0;
		ff0[im] = f0;
		ff1[im] = f1;
		CN0im[im] = CentNear0;
		CN1im[im] = CentNear1;
		FC0im[im] = FarCent0;
		FC1im[im] = FarCent1;

		Movieframes[im].centerX = f1*Keyframes[k-1].centerX + f0*Keyframes[k].centerX;
		Movieframes[im].centerY = f1*Keyframes[k-1].centerY + f0*Keyframes[k].centerY;
		Movieframes[im].centerZ = f1*Keyframes[k-1].centerZ + f0*Keyframes[k].centerZ;
		Movieframes[im].upX     = f1*Keyframes[k-1].upX     + f0*Keyframes[k].upX;
		Movieframes[im].upY     = f1*Keyframes[k-1].upY     + f0*Keyframes[k].upY;
		Movieframes[im].upZ     = f1*Keyframes[k-1].upZ     + f0*Keyframes[k].upZ;

		Movieframes[im].my_near = (float)1.;  //  Only set AFTER eye and center are smoothed
		Movieframes[im].my_far  = (float)1.;  //  Only set AFTER eye and center are smoothed
		Movieframes[im].nplanes = f1*Keyframes[k-1].nplanes + f0*Keyframes[k].nplanes;
		Movieframes[im].opacity = f1*Keyframes[k-1].opacity + f0*Keyframes[k].opacity;
		Movieframes[im].fovangle =f1*Keyframes[k-1].fovangle+ f0*Keyframes[k].fovangle;
		Movieframes[im].softdistance = ( f1 * Keyframes[k-1].softdistance +
										  f0 * Keyframes[k  ].softdistance   );

		MovieSluts[im].iNAnots = KeySluts[k-1].iNAnots;
		for(inot=0; inot<MovieSluts[im].iNAnots; inot++) {
			MovieSluts[im].iAind[inot] = (int)(f1 * (float)KeySluts[k-1].iAind[inot] + 
											   f0 * (float)KeySluts[k  ].iAind[inot] +  0.5);
			MovieSluts[im].fAlp[inot]  = f1 * KeySluts[k-1].fAlp[inot]   +  f0 * KeySluts[k].fAlp[inot];
		}

		MovieSluts[im].iNCnots = KeySluts[k-1].iNCnots;
		for(inot=0; inot<MovieSluts[im].iNCnots; inot++) {
			MovieSluts[im].iCind[inot] = (int)(f1 * (float)KeySluts[k-1].iCind[inot] + 
											   f0 * (float)KeySluts[k  ].iCind[inot] +  0.5);
			MovieSluts[im].fRed[inot]  = f1 * KeySluts[k-1].fRed[inot]   +  f0 * KeySluts[k].fRed[inot];
			MovieSluts[im].fGrn[inot]  = f1 * KeySluts[k-1].fGrn[inot]   +  f0 * KeySluts[k].fGrn[inot];
			MovieSluts[im].fBlu[inot]  = f1 * KeySluts[k-1].fBlu[inot]   +  f0 * KeySluts[k].fBlu[inot];
		}

		for(jc=0; jc<4; jc++) { for(ic=0; ic<6; ic++) {
		Movieframes[im].auxclip[ic][jc] = f1*Keyframes[k-1].auxclip[ic][jc] +
										  f0*Keyframes[k  ].auxclip[ic][jc];
		}}

        fnrpk = Keyframes[k].nrpk;
        pi2 = (float)2. * (float)3.141592654;

        angle =  f0 * fnrpk * pi2;  gen_eye(xyz0, angle, &Keyframes[k-1]);
        angle = -f1 * fnrpk * pi2;  gen_eye(xyz1, angle, &Keyframes[k  ]);

        Movieframes[im].eyeX = f1 * xyz0[0]  +  f0 * xyz1[0];
        Movieframes[im].eyeY = f1 * xyz0[1]  +  f0 * xyz1[1];
        Movieframes[im].eyeZ = f1 * xyz0[2]  +  f0 * xyz1[2];

        im++;
      }
    }
    nmovieframes = im;

	// MUST smooth path BEFORE setting near and far clipping planes.
    smooth_path();
    smooth_path();

	for(im = 0; im < nmovieframes; im++) {
		x = Movieframes[im].eyeX - Movieframes[im].centerX;
		y = Movieframes[im].eyeY - Movieframes[im].centerY;
		z = Movieframes[im].eyeZ - Movieframes[im].centerZ;
		distance = (float)sqrt((double)(x*x + y*y + z*z));

		f0 = ff0[im];
		f1 = ff1[im];

		CurrentCentNear = f1*CN0im[im] + f0*CN1im[im];
		if(CurrentCentNear > (float)0.9*distance) { CurrentCentNear = (float)0.9*distance; }

        Movieframes[im].my_near = distance - CurrentCentNear;
        Movieframes[im].my_far  = distance + f1*FC0im[im]  + f0*FC1im[im];
/*
		fperr = fopen("KeysToHvrIn.out", "a+");
		fprintf(fperr, "%f %f %f %f %f %f %f\n",
			f1,CentNear0,f0,CentNear1,CurrentCentNear,distance,Movieframes[im].my_near);
		fclose(fperr);
*/
	}

	return (nmovieframes);
}

int GetTail(char *str, char *tail)
{
	int i, i0, iLen;
	iLen = strlen(str);
	for(i=0; i<iLen; i++) { if(str[i] == '/'  ||  str[i] == '\\') { i0 = i+1; } }
	for(i=i0; i<iLen; i++) { tail[i-i0] = str[i]; }

	return (iLen-i0);
}

void smooth_one( float *xx )
{
  int i, j, ip;
  float xxm, xx0, xxp;

  if(iwrap == 0) {
    for(j=0; j<10; j++) {
      xxm = xx[0];
      for(i=1; i<nmovieframes-1; i++) {
        xx0 = *(float*)((char*)xx+sizeof(Rp)*i);
        xxp = *(float*)((char*)xx+sizeof(Rp)*(i+1));
        *(float*)((char*)xx+sizeof(Rp)*i) = (float)0.25 * (xxm + xxp)  +  (float)0.5 * xx0;
        xxm = xx0;
      }
    }
  } else {
    for(j=0; j<10; j++) {
      xxm = *(float*)((char*)xx+sizeof(Rp)*(nmovieframes-1));
      for(i=0; i<nmovieframes; i++) {
        ip = i+1; if(ip >= nmovieframes) { ip = 0; }
        xx0 = *(float*)((char*)xx+sizeof(Rp)*i );
        xxp = *(float*)((char*)xx+sizeof(Rp)*ip);
        *(float*)((char*)xx+sizeof(Rp)*i) = (float)0.25 * (xxm + xxp)  +  (float)0.5 * xx0;
        xxm = xx0;
      }
    }
  }
}

void smooth_path( void )
{
    int ic, jc;
    void misc_recompile_all(), misc_redraw_finder();

    smooth_one( &Movieframes[0].eyeX );
    smooth_one( &Movieframes[0].eyeY );
    smooth_one( &Movieframes[0].eyeZ );
    smooth_one( &Movieframes[0].centerX );
    smooth_one( &Movieframes[0].centerY );
    smooth_one( &Movieframes[0].centerZ );
    smooth_one( &Movieframes[0].upX );
    smooth_one( &Movieframes[0].upY );
    smooth_one( &Movieframes[0].upZ );
    smooth_one( &Movieframes[0].my_near );
    smooth_one( &Movieframes[0].my_far );
    smooth_one( &Movieframes[0].nplanes );
    smooth_one( &Movieframes[0].opacity );
    smooth_one( &Movieframes[0].fovangle );
    smooth_one( &Movieframes[0].softdistance );
    for(jc=0; jc<4; jc++) { for(ic=0; ic<6; ic++) {
      smooth_one( &Movieframes[0].auxclip[ic][jc] );
    }}

}

