// Edit_View.cpp : implementation file
//
////////////////////////////////////////////////////////////////////////////////
//
// Author:   David H. Porter
//           Laboratory for Computational Science & Engineering
//           University of Minnesota
//
////////////////////////////////////////////////////////////////////////////////
/***************************************************************************************
Program:  HVR_SERVER  --  Hierarchical Volume Rendering Component Object Moduel
Copyright (C) 2002  David H. Porter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************************/

#include "stdafx.h"
#include "hvr_mfc.h"
#include "hvr_mfcDlg.h"
#include "Edit_View.h"
#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

static CHvr_mfcDlg *pMainDlg;

/////////////////////////////////////////////////////////////////////////////
// CEdit_View dialog


CEdit_View::CEdit_View(CWnd* pParent /*=NULL*/)
	: CDialog(CEdit_View::IDD, pParent)
{
	//{{AFX_DATA_INIT(CEdit_View)
	m_fCenX = 0.0f;
	m_fCenY = 0.0f;
	m_fCenZ = 0.0f;
	m_fEyeX = 0.0f;
	m_fEyeY = 0.0f;
	m_fEyeZ = 0.0f;
	m_fUpX = 0.0f;
	m_fUpY = 0.0f;
	m_fUpZ = 0.0f;
	m_fNearFar = 0.0f;
	m_fNearLim = 0.0f;
	//}}AFX_DATA_INIT
}


void CEdit_View::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CEdit_View)
	DDX_Text(pDX, IDC_EV_EDIT_CENT_X, m_fCenX);
	DDX_Text(pDX, IDC_EV_EDIT_CENT_Y, m_fCenY);
	DDX_Text(pDX, IDC_EV_EDIT_CENT_Z, m_fCenZ);
	DDX_Text(pDX, IDC_EV_EDIT_EYE_X, m_fEyeX);
	DDX_Text(pDX, IDC_EV_EDIT_EYE_Y, m_fEyeY);
	DDX_Text(pDX, IDC_EV_EDIT_EYE_Z, m_fEyeZ);
	DDX_Text(pDX, IDC_EV_EDIT_UP_X, m_fUpX);
	DDX_Text(pDX, IDC_EV_EDIT_UP_Y, m_fUpY);
	DDX_Text(pDX, IDC_EV_EDIT_UP_Z, m_fUpZ);
	DDX_Text(pDX, IDC_EV_EDIT_NEAR_FAR, m_fNearFar);
	DDX_Text(pDX, IDC_EV_EDIT_NEAR_LIMIT, m_fNearLim);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CEdit_View, CDialog)
	//{{AFX_MSG_MAP(CEdit_View)
	ON_BN_CLICKED(IDC_EV_BUTTON_EYE_X, OnEvButtonEyeX)
	ON_BN_CLICKED(IDC_EV_BUTTON_EYE_Y, OnEvButtonEyeY)
	ON_BN_CLICKED(IDC_EV_BUTTON_EYE_Z, OnEvButtonEyeZ)
	ON_BN_CLICKED(IDC_EV_BUTTON_UP_X, OnEvButtonUpX)
	ON_BN_CLICKED(IDC_EV_BUTTON_UP_Y, OnEvButtonUpY)
	ON_BN_CLICKED(IDC_EV_BUTTON_UP_Z, OnEvButtonUpZ)
	ON_BN_CLICKED(IDC_EV_BUTTON_ZERO_CENT, OnEvButtonZeroCent)
	ON_BN_CLICKED(IDC_EV_BUTTON_APPLY, OnEvButtonApply)
	ON_BN_CLICKED(IDC_EV_BUTTON_SAVE, OnEvButtonSave)
	ON_BN_CLICKED(IDC_EV_BUTTON_LOAD, OnEvButtonLoad)
	ON_BN_CLICKED(IDC_EV_BUTTON_RESET, OnEvButtonReset)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CEdit_View message handlers

// The next 3 methods reset the Eye-Center vector (m_fEyeX, m_fEyeY, m_fEyeZ)
// to be along the +X, +Y, or +Z direction while preserving the distance from eye to center.
void CEdit_View::OnEvButtonEyeX() 
{
	double distance = sqrt((double)(m_fEyeX*m_fEyeX+m_fEyeY*m_fEyeY+m_fEyeZ*m_fEyeZ));
	m_fEyeX = (float)distance;
	m_fEyeY = (float)0.;
	m_fEyeZ = (float)0.;
	UpdateData(FALSE);
}

void CEdit_View::OnEvButtonEyeY() 
{
	double distance = sqrt((double)(m_fEyeX*m_fEyeX+m_fEyeY*m_fEyeY+m_fEyeZ*m_fEyeZ));
	m_fEyeX = (float)0.;
	m_fEyeY = (float)distance;
	m_fEyeZ = (float)0.;
	UpdateData(FALSE);
}

void CEdit_View::OnEvButtonEyeZ() 
{
	double distance = sqrt((double)(m_fEyeX*m_fEyeX+m_fEyeY*m_fEyeY+m_fEyeZ*m_fEyeZ));
	m_fEyeX = (float)0.;
	m_fEyeY = (float)0.;
	m_fEyeZ = (float)distance;
	UpdateData(FALSE);
}

// The next 3 methods reset the Up vector (m_fUpX, m_fUpY, m_fUpZ)
// to be along the +X, +Y, or +Z direction.
void CEdit_View::OnEvButtonUpX() 
{
	m_fUpX = (float)1.;
	m_fUpY = (float)0.;
	m_fUpZ = (float)0.;
	UpdateData(FALSE);
}

void CEdit_View::OnEvButtonUpY() 
{
	m_fUpX = (float)0.;
	m_fUpY = (float)1.;
	m_fUpZ = (float)0.;
	UpdateData(FALSE);
}

void CEdit_View::OnEvButtonUpZ() 
{
	m_fUpX = (float)0.;
	m_fUpY = (float)0.;
	m_fUpZ = (float)1.;
	UpdateData(FALSE);
}

// Reset the center vector (m_fCenX, m_fCenY, m_fCenZ) to be at the center of
// the volume (0, 0, 0).
void CEdit_View::OnEvButtonZeroCent() 
{
	m_fCenX = (float)0.;
	m_fCenY = (float)0.;
	m_fCenZ = (float)0.;
	UpdateData(FALSE);
}

// Get View parameters from serever object
void CEdit_View::GetViewFromServer()
{
//	struct Renderparams srpKeyInfo;

	pMainDlg->StopRendering();
	pChvrMainPort[0]->SaveKey(&m_srpKeyInfo);

	m_fCenX = m_srpKeyInfo.centerX;
	m_fCenY = m_srpKeyInfo.centerY;
	m_fCenZ = m_srpKeyInfo.centerZ;
	m_fEyeX = m_srpKeyInfo.eyeX - m_srpKeyInfo.centerX;
	m_fEyeY = m_srpKeyInfo.eyeY - m_srpKeyInfo.centerY;
	m_fEyeZ = m_srpKeyInfo.eyeZ - m_srpKeyInfo.centerZ;
	m_fUpX = m_srpKeyInfo.upX;
	m_fUpY = m_srpKeyInfo.upY;
	m_fUpZ = m_srpKeyInfo.upZ;

	double distance = sqrt((double)(m_fEyeX*m_fEyeX+m_fEyeY*m_fEyeY+m_fEyeZ*m_fEyeZ));

	m_fNearFar = m_srpKeyInfo.my_far - (float)distance;
	m_fNearLim = (float)distance - m_srpKeyInfo.my_near;
}

// Apply new view parameters to rendereing object
void CEdit_View::OnEvButtonApply() 
{
	pMainDlg->StopRendering();
	UpdateData(TRUE);

	// Make sure that Up and Eye-Center vectors are not parallel
	float doteu = m_fEyeX * m_fUpX  + m_fEyeY * m_fUpY  + m_fEyeZ * m_fUpZ;
	float dotuu = m_fUpX  * m_fUpX  + m_fUpY  * m_fUpY  + m_fUpZ  * m_fUpZ;
	float dotee = m_fEyeX * m_fEyeX + m_fEyeY * m_fEyeY + m_fEyeZ * m_fEyeZ;
	float doteueu = doteu * doteu;
	float doteeuu = dotee * dotuu;
	float diff = doteeuu - doteueu;
	if(diff < 0.00001 * doteeuu) {
		MessageBox("Impossible view: Up and Eye-Center vectors are parallel.");
		m_bWasApplied = FALSE;
		return;
	}

	// Apply limits on near and far clipping
	double distance = sqrt((double)(m_fEyeX*m_fEyeX+m_fEyeY*m_fEyeY+m_fEyeZ*m_fEyeZ));
	float  dist9    = (float)(0.9 * distance);
	if(dist9 > (float)10.0) { dist9 = (float)10.0; }
	if(m_fNearLim < (float)1.e-4) { m_fNearLim = (float)1.e-4;  UpdateData(FALSE); }
	if(m_fNearFar < (float)1.e-4) { m_fNearFar = (float)1.e-4;  UpdateData(FALSE); }
	if(m_fNearLim > dist9       ) { m_fNearLim = dist9;         UpdateData(FALSE); }
	if(m_fNearFar > (float)10.0 ) { m_fNearFar = (float)10.0;   UpdateData(FALSE); }

	// Set members of Renderparams stucture to reflect new view parameters
	m_srpKeyInfo.centerX = m_fCenX;
	m_srpKeyInfo.centerY = m_fCenY;
	m_srpKeyInfo.centerZ = m_fCenZ;
	m_srpKeyInfo.eyeX    = m_fEyeX + m_fCenX;
	m_srpKeyInfo.eyeY    = m_fEyeY + m_fCenY;
	m_srpKeyInfo.eyeZ    = m_fEyeZ + m_fCenZ;
	m_srpKeyInfo.upX     = m_fUpX;
	m_srpKeyInfo.upY     = m_fUpY;
	m_srpKeyInfo.upZ     = m_fUpZ;
	m_srpKeyInfo.my_far  = (float)distance + m_fNearFar;
	m_srpKeyInfo.my_near = (float)distance - m_fNearLim;

	// Send Renderparams struct to render object and render with new parameters
	pChvrMainPort[0]->RestorKey(&m_srpKeyInfo);
	pMainDlg->DoQualityFrame();

	m_bWasApplied = TRUE;
}

// Save view parameters to a file
void CEdit_View::OnEvButtonSave() 
{
	FILE *fp;
	CFileDialog dlgFile(FALSE);

	UpdateData(TRUE);

	dlgFile.m_ofn.lpstrFilter = "HVR View Files (*.view)\0*.view\0\0";

	if(dlgFile.DoModal() == IDOK) {	
		CString csFile = dlgFile.GetPathName();
		if((LPCTSTR)csFile != "") {
			fp = fopen((LPCTSTR)csFile, "w");
			fprintf(fp, "Center      %f  %f  %f\n", m_fCenX, m_fCenY, m_fCenZ);
			fprintf(fp, "Eye-Center  %f  %f  %f\n", m_fEyeX, m_fEyeY, m_fEyeZ);
			fprintf(fp, "Up          %f  %f  %f\n", m_fUpX,  m_fUpY,  m_fUpZ);
			fprintf(fp, "FarClip-Cent    %f\n", m_fNearFar);
			fprintf(fp, "Cent-NearClip   %f\n", m_fNearLim);
			fclose(fp);
		}
	}
}

// Get view parameters from a file
void CEdit_View::OnEvButtonLoad() 
{
	FILE *fp;
	CFileDialog dlgFile(TRUE);

	dlgFile.m_ofn.lpstrFilter = "HVR View Files (*.view)\0*.view\0\0";

	if(dlgFile.DoModal() == IDOK) {	
		CString csFile = dlgFile.GetPathName();
		if((LPCTSTR)csFile != "") {
			fp = fopen((LPCTSTR)csFile, "r");
			fscanf(fp, "Center      %f  %f  %f\n", &m_fCenX, &m_fCenY, &m_fCenZ);
			fscanf(fp, "Eye-Center  %f  %f  %f\n", &m_fEyeX, &m_fEyeY, &m_fEyeZ);
			fscanf(fp, "Up          %f  %f  %f\n", &m_fUpX,  &m_fUpY,  &m_fUpZ);
			fscanf(fp, "FarClip-Cent    %f\n", &m_fNearFar);
			fscanf(fp, "Cent-NearClip   %f\n", &m_fNearLim);
			fclose(fp);
			UpdateData(FALSE);

			OnEvButtonApply();
		}
	}

}

void CEdit_View::OnOK() 
{
	OnEvButtonApply();
	if(!m_bWasApplied) {
		MessageBox("Impossible view: Up and Eye-Center vectors are parallel.");
		return;
	}
	
	CDialog::OnOK();
}

void CEdit_View::OnEvButtonReset() 
{
	m_fCenX = (float)0.;
	m_fCenY = (float)0.;
	m_fCenZ = (float)0.;
	m_fEyeX = (float)0.;
	m_fEyeY = (float)0.;
	m_fEyeZ = (float)10.;
	m_fUpX = (float)0.;
	m_fUpY = (float)1.;
	m_fUpZ = (float)0.;
	m_fNearFar = (float)10.;
	m_fNearLim = (float)9.0;

	UpdateData(FALSE);
		
}

void CEdit_View::SetPP(CHvr_mfcDlg *pp)
{
	// set the pointer to the parent class
	pMainDlg = pp;
}
