// EditSubregion.cpp : implementation file
//
////////////////////////////////////////////////////////////////////////////////
//
// Author:   David H. Porter
//           Laboratory for Computational Science & Engineering
//           University of Minnesota
//
////////////////////////////////////////////////////////////////////////////////
/***************************************************************************************
Program:  HVR_SERVER  --  Hierarchical Volume Rendering Component Object Moduel
Copyright (C) 2002  David H. Porter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************************/

#include "stdafx.h"
#include "hvr_mfc.h"
#include "hvr_mfcDlg.h"
#include "EditSubregion.h"
#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

static CHvr_mfcDlg *pMainDlg;

/////////////////////////////////////////////////////////////////////////////
// CEditSubregion dialog


CEditSubregion::CEditSubregion(CWnd* pParent /*=NULL*/)
	: CDialog(CEditSubregion::IDD, pParent)
{
	//{{AFX_DATA_INIT(CEditSubregion)
	m_bUseAuxClips = FALSE;
	m_fXmax = 0.0f;
	m_fXmin = 0.0f;
	m_fYmax = 0.0f;
	m_fYmin = 0.0f;
	m_fZmax = 0.0f;
	m_fZmin = 0.0f;
	m_iXcenter = 0;
	m_iXsize = 0;
	m_iYcenter = 0;
	m_iYsize = 0;
	m_iZcenter = 0;
	m_iZsize = 0;
	//}}AFX_DATA_INIT
}


void CEditSubregion::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CEditSubregion)
	DDX_Check(pDX, IDC_CHECK_USE_AUX_CLIPS, m_bUseAuxClips);
	DDX_Text(pDX, IDC_EDIT_XMAX, m_fXmax);
	DDX_Text(pDX, IDC_EDIT_XMIN, m_fXmin);
	DDX_Text(pDX, IDC_EDIT_YMAX, m_fYmax);
	DDX_Text(pDX, IDC_EDIT_YMIN, m_fYmin);
	DDX_Text(pDX, IDC_EDIT_ZMAX, m_fZmax);
	DDX_Text(pDX, IDC_EDIT_ZMIN, m_fZmin);
	DDX_Slider(pDX, IDC_SLIDER_X_CENT, m_iXcenter);
	DDX_Slider(pDX, IDC_SLIDER_X_SIZE, m_iXsize);
	DDX_Slider(pDX, IDC_SLIDER_Y_CENT, m_iYcenter);
	DDX_Slider(pDX, IDC_SLIDER_Y_SIZE, m_iYsize);
	DDX_Slider(pDX, IDC_SLIDER_Z_CENT, m_iZcenter);
	DDX_Slider(pDX, IDC_SLIDER_Z_SIZE, m_iZsize);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CEditSubregion, CDialog)
	//{{AFX_MSG_MAP(CEditSubregion)
	ON_BN_CLICKED(IDC_BUTTON_APPLY, OnButtonApply)
	ON_NOTIFY(NM_RELEASEDCAPTURE, IDC_SLIDER_X_CENT, OnReleasedcaptureSliderXCent)
	ON_NOTIFY(NM_RELEASEDCAPTURE, IDC_SLIDER_Y_CENT, OnReleasedcaptureSliderYCent)
	ON_NOTIFY(NM_RELEASEDCAPTURE, IDC_SLIDER_Z_CENT, OnReleasedcaptureSliderZCent)
	ON_NOTIFY(NM_RELEASEDCAPTURE, IDC_SLIDER_X_SIZE, OnReleasedcaptureSliderXSize)
	ON_NOTIFY(NM_RELEASEDCAPTURE, IDC_SLIDER_Y_SIZE, OnReleasedcaptureSliderYSize)
	ON_NOTIFY(NM_RELEASEDCAPTURE, IDC_SLIDER_Z_SIZE, OnReleasedcaptureSliderZSize)
	ON_BN_CLICKED(IDC_BUTTON_RESET_LIMITS, OnButtonResetLimits)
	ON_BN_CLICKED(IDC_BUTTON_CENTER_EYE_LIMITS, OnButtonCenterEyeLimits)
	ON_BN_CLICKED(IDC_CHECK_USE_AUX_CLIPS, OnCheckUseAuxClips)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CEditSubregion message handlers

void CEditSubregion::OnButtonApply() 
{

	UpdateData(TRUE);

	// Set Sliders from XYZ Limits
	double size, center;

	// Do X sliders
	center = 0.5 * (double)(m_fXmax + m_fXmin);
	size  = (double)(m_fXmax - m_fXmin);
	m_iXcenter = 50 + (int)(50.5 * center);
	m_iXsize = 66 + (int)(0.5 + log((double)size)/(log(10.0) * 0.03333333));
	if(m_iXcenter <   0) { m_iXcenter =   0; }
	if(m_iXcenter > 100) { m_iXcenter = 100; }
	if(m_iXsize <   0) { m_iXsize =   0; }
	if(m_iXsize > 100) { m_iXsize = 100; }
	// Do Y sliders
	center = 0.5 * (double)(m_fYmax + m_fYmin);
	size  = (double)(m_fYmax - m_fYmin);
	m_iYcenter = 50 + (int)(50.5 * center);
	m_iYsize = 66 + (int)(0.5 + log((double)size)/(log(10.0) * 0.03333333));
	if(m_iYcenter <   0) { m_iYcenter =   0; }
	if(m_iYcenter > 100) { m_iYcenter = 100; }
	if(m_iYsize <   0) { m_iYsize =   0; }
	if(m_iYsize > 100) { m_iYsize = 100; }
	// Do Z sliders
	center = 0.5 * (double)(m_fZmax + m_fZmin);
	size  = (double)(m_fZmax - m_fZmin);
	m_iZcenter = 50 + (int)(50.5 * center);
	m_iZsize = 66 + (int)(0.5 + log((double)size)/(log(10.0) * 0.03333333));
	if(m_iZcenter <   0) { m_iZcenter =   0; }
	if(m_iZcenter > 100) { m_iZcenter = 100; }
	if(m_iZsize <   0) { m_iZsize =   0; }
	if(m_iZsize > 100) { m_iZsize = 100; }

	//Apply New Limits to HVR Server
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();
}

void CEditSubregion::SetPP(CHvr_mfcDlg *pp)
{
	// set the pointer to the parent class
	pMainDlg = pp;

	// Since this is done only oncde at startup, set initial values
	m_bUseAuxClips = FALSE;
	m_fXmin = (float)(-1.0);
	m_fXmax = (float)( 1.0);
	m_fYmin = (float)(-1.0);
	m_fYmax = (float)( 1.0);
	m_fZmin = (float)(-1.0);
	m_fZmax = (float)( 1.0);


	double center, size;
	// Do X sliders
	center = 0.5 * (double)(m_fXmax + m_fXmin);
	size  = (double)(m_fXmax - m_fXmin);
	m_iXcenter = 50 + (int)(50.5 * center);
	m_iXsize = 66 + (int)(0.5 + log((double)size)/(log(10.0) * 0.03333333));
	if(m_iXcenter <   0) { m_iXcenter =   0; }
	if(m_iXcenter > 100) { m_iXcenter = 100; }
	if(m_iXsize <   0) { m_iXsize =   0; }
	if(m_iXsize > 100) { m_iXsize = 100; }
	// Do Y sliders
	center = 0.5 * (double)(m_fYmax + m_fYmin);
	size  = (double)(m_fYmax - m_fYmin);
	m_iYcenter = 50 + (int)(50.5 * center);
	m_iYsize = 66 + (int)(0.5 + log((double)size)/(log(10.0) * 0.03333333));
	if(m_iYcenter <   0) { m_iYcenter =   0; }
	if(m_iYcenter > 100) { m_iYcenter = 100; }
	if(m_iYsize <   0) { m_iYsize =   0; }
	if(m_iYsize > 100) { m_iYsize = 100; }
	// Do Z sliders
	center = 0.5 * (double)(m_fZmax + m_fZmin);
	size  = (double)(m_fZmax - m_fZmin);
	m_iZcenter = 50 + (int)(50.5 * center);
	m_iZsize = 66 + (int)(0.5 + log((double)size)/(log(10.0) * 0.03333333));
	if(m_iZcenter <   0) { m_iZcenter =   0; }
	if(m_iZcenter > 100) { m_iZcenter = 100; }
	if(m_iZsize <   0) { m_iZsize =   0; }
	if(m_iZsize > 100) { m_iZsize = 100; }

}

void CEditSubregion::OnReleasedcaptureSliderXCent(NMHDR* pNMHDR, LRESULT* pResult) 
{
	UpdateData(TRUE);
	float size;
	size = (float)exp(log(10.0) * 0.033 * (double)(m_iXsize - 66));
	m_fXmin = (float)0.02 * (float)(m_iXcenter - 50) - (float)0.5 * size;
	m_fXmax = (float)0.02 * (float)(m_iXcenter - 50) + (float)0.5 * size;
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();

	*pResult = 0;
}

void CEditSubregion::OnReleasedcaptureSliderYCent(NMHDR* pNMHDR, LRESULT* pResult) 
{
	UpdateData(TRUE);
	float size;
	size = (float)exp(log(10.0) * 0.03333333 * (double)(m_iYsize - 66));
	m_fYmin = (float)0.02 * (float)(m_iYcenter - 50) - (float)0.5 * size;
	m_fYmax = (float)0.02 * (float)(m_iYcenter - 50) + (float)0.5 * size;
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();

	*pResult = 0;
}

void CEditSubregion::OnReleasedcaptureSliderZCent(NMHDR* pNMHDR, LRESULT* pResult) 
{
	UpdateData(TRUE);
	float size;
	size = (float)exp(log(10.0) * 0.03333333 * (double)(m_iZsize - 66));
	m_fZmin = (float)0.02 * (float)(m_iZcenter - 50) - (float)0.5 * size;
	m_fZmax = (float)0.02 * (float)(m_iZcenter - 50) + (float)0.5 * size;
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();
	
	*pResult = 0;
}

void CEditSubregion::OnReleasedcaptureSliderXSize(NMHDR* pNMHDR, LRESULT* pResult) 
{
	UpdateData(TRUE);
	float size;
	size = (float)exp(log(10.0) * 0.03333333 * (double)(m_iXsize - 66));
	m_fXmin = (float)0.02 * (float)(m_iXcenter - 50) - (float)0.5 * size;
	m_fXmax = (float)0.02 * (float)(m_iXcenter - 50) + (float)0.5 * size;
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();

	*pResult = 0;
}

void CEditSubregion::OnReleasedcaptureSliderYSize(NMHDR* pNMHDR, LRESULT* pResult) 
{
	UpdateData(TRUE);
	float size;
	size = (float)exp(log(10.0) * 0.03333333 * (double)(m_iYsize - 66));
	m_fYmin = (float)0.02 * (float)(m_iYcenter - 50) - (float)0.5 * size;
	m_fYmax = (float)0.02 * (float)(m_iYcenter - 50) + (float)0.5 * size;
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();

	*pResult = 0;
}

void CEditSubregion::OnReleasedcaptureSliderZSize(NMHDR* pNMHDR, LRESULT* pResult) 
{
	UpdateData(TRUE);
	float size;
	size = (float)exp(log(10.0) * 0.03333333 * (double)(m_iZsize - 66));
	m_fZmin = (float)0.02 * (float)(m_iZcenter - 50) - (float)0.5 * size;
	m_fZmax = (float)0.02 * (float)(m_iZcenter - 50) + (float)0.5 * size;
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);
	ApplyLimits();

	*pResult = 0;
}

void CEditSubregion::OnButtonResetLimits() 
{
	m_bUseAuxClips = TRUE;
	m_iXcenter = 50;
	m_iYcenter = 50;
	m_iZcenter = 50;
	m_iXsize   = 66;
	m_iYsize   = 66;
	m_iZsize   = 66;
	m_fXmin = (float)(-1.0);
	m_fXmax = (float)( 1.0);
	m_fYmin = (float)(-1.0);
	m_fYmax = (float)( 1.0);
	m_fZmin = (float)(-1.0);
	m_fZmax = (float)( 1.0);
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);

	// Reset sliders and apply to HVR server
	OnButtonApply();
}

void CEditSubregion::ApplyLimits()
{
	// Get auxclip values from from
	UpdateData(TRUE);

	// Stop any QH renderings and wait for Rendering Quality parameters to be reset
	pMainDlg->StopRendering();

	// Get current rendering parameters from local HVR server
	Renderparams rp;
	pMainDlg->pChvrMainPort[0]->SaveKey(&rp);

	// set new auxclip values in renderparams struct
	int i;
	if(m_bUseAuxClips == TRUE) {
		for(i=0; i<6; i++) { rp.ifauxclip[i]  = 2; }
		rp.auxclip[0][3] = m_fXmin;
		rp.auxclip[1][3] = m_fXmax;
		rp.auxclip[2][3] = m_fYmin;
		rp.auxclip[3][3] = m_fYmax;
		rp.auxclip[4][3] = m_fZmin;
		rp.auxclip[5][3] = m_fZmax;
	} else {
		for(i=0; i<6; i++) { rp.ifauxclip[i] = 0; }
	}

	// send new rendering parameters to local HVR serve
	// and set it rendering
	pMainDlg->pChvrMainPort[0]->RestorKey(&rp);
	pMainDlg->DoQualityFrame();
}

void CEditSubregion::OnButtonCenterEyeLimits() 
{
	// Stop any QH renderings and wait for Rendering Quality parameters to be reset
	pMainDlg->StopRendering();

	// Get current rendering parameters from local HVR server
	Renderparams rp;
	pMainDlg->pChvrMainPort[0]->SaveKey(&rp);

	// Calculate Limits Based on Center and Eye vectors
	double dx = (double)(rp.centerX - rp.eyeX);
	double dy = (double)(rp.centerY - rp.eyeY);
	double dz = (double)(rp.centerZ - rp.eyeZ);
	float size = (float)(0.2 * sqrt(dx*dx + dy*dy + dz*dz));
	m_fXmin = rp.centerX - size;
	m_fXmax = rp.centerX + size;
	m_fYmin = rp.centerY - size;
	m_fYmax = rp.centerY + size;
	m_fZmin = rp.centerZ - size;
	m_fZmax = rp.centerZ + size;

	// Set GUI Limits to these values
	m_bUseAuxClips = TRUE;
	UpdateData(FALSE);

	// Set Sliders to Match and Apply to HVR Server
	OnButtonApply();
}

void CEditSubregion::OnCheckUseAuxClips() 
{
	UpdateData(TRUE);
	ApplyLimits();
}
