// EditLUT.cpp : implementation file
//
////////////////////////////////////////////////////////////////////////////////
//
// Author:   David H. Porter
//           Laboratory for Computational Science & Engineering
//           University of Minnesota
//
////////////////////////////////////////////////////////////////////////////////
/***************************************************************************************
Program:  HVR_SERVER  --  Hierarchical Volume Rendering Component Object Moduel
Copyright (C) 2002  David H. Porter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************************/

#include "stdafx.h"
#include "hvr_mfc.h"
#include "hvr_mfcDlg.h"
#include "EditLUT.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CEditLUT dialog

static CHvr_mfcDlg *pMainDlg;

CEditLUT::CEditLUT(CWnd* pParent /*=NULL*/)
	: CDialog(CEditLUT::IDD, pParent)
{
	//{{AFX_DATA_INIT(CEditLUT)
	m_csKnotIndex = _T("");
	//}}AFX_DATA_INIT
}


void CEditLUT::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CEditLUT)
	DDX_Text(pDX, IDC_LUT_EDIT_KNOT, m_csKnotIndex);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CEditLUT, CDialog)
	//{{AFX_MSG_MAP(CEditLUT)
	ON_WM_LBUTTONDOWN()
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONUP()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_BN_CLICKED(IDC_LUT_BUTTON_SAVE, OnLutButtonSave)
	ON_BN_CLICKED(IDC_LUT_BUTTON_LOAD, OnLutButtonLoad)
	ON_WM_SHOWWINDOW()
	ON_WM_PAINT()
	ON_BN_CLICKED(IDC_LUT_DELETE_KNOT, OnLutDeleteKnot)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CEditLUT message handlers

void CEditLUT::CreateOnce()
{
	if(m_iAmCreated == 0) {
		Create(IDD_DIALOG_EDiT_LUT, this);
		SetWindowPos(&wndTopMost, 10, 550, 562, 410, SWP_DRAWFRAME);
		m_iAmCreated = 1;
//		InitPlots();
	}
	ShowWindow(SW_SHOW);
//	DrawPlots();
	Invalidate();
}

void CEditLUT::InitPlots()
{
	m_iChosenNot    = -1;
	m_iAlpChosenNot = -1;
	m_iAlpValCosenNot = -1;
	m_iXrgb =  20;
	m_iYrgb = 165;
	m_iXalp =  20;
	m_iYalp = 335;
	m_iXsize = 512;
	m_iYsize = 128;
	m_crRGB.SetRect(m_iXrgb, m_iYrgb-m_iYsize, m_iXrgb+m_iXsize, m_iYrgb);
	m_crALP.SetRect(m_iXalp, m_iYalp-m_iYsize, m_iXalp+m_iXsize, m_iYalp);

	m_iNrgbNots = 1;
	m_iNalpNots = 1;
	m_rgbNots[0].ind = 0;
	m_rgbNots[0].red = 0;
	m_rgbNots[0].grn = 0;
	m_rgbNots[0].blu = 0;

	m_rgbNots[1].ind = 255;
	m_rgbNots[1].red = 255;
	m_rgbNots[1].grn = 255;
	m_rgbNots[1].blu = 255;

	m_alpNots[0].ind = 0;
	m_alpNots[0].alp = 0;
	m_alpNots[1].ind = 255;
	m_alpNots[1].alp = 255;

	m_csKnotIndex = " ";
	m_iKnotIndexRGB = -1;
	m_iKnotIndexALP = -1;
}

void CEditLUT::DrawPlots()
{
//	static CPaintDC dc(this); // device context for painting
	CPaintDC dc(this); // device context for painting

	CBrush RectBrush(RGB(0,0,0));
	RECT myRect;

	myRect.bottom = m_iYrgb + 10;;
	myRect.top    = m_iYrgb - m_iYsize - 10;;
	myRect.left   = m_iXrgb - 10;
	myRect.right  = m_iXrgb + m_iXsize + 10;
//	dc.FillRect(&myRect, &RectBrush);

	myRect.bottom = m_iYalp + 10;;
	myRect.top    = m_iYalp - m_iYsize - 10;;
	myRect.left   = m_iXalp - 10;
	myRect.right  = m_iXalp + m_iXsize + 10;
//	dc.FillRect(&myRect, &RectBrush);

//	CPen  BlackPen(PS_SOLID, 1, RGB(  0,   0,   0));
	CPen LimitsPen(PS_SOLID, 1, RGB( 80,  80,  80));
	CPen    redPen(PS_SOLID, 2, RGB(255,   0,   0));
	CPen    grnPen(PS_SOLID, 2, RGB(  0, 255,   0));
	CPen    bluPen(PS_SOLID, 2, RGB(  0,   0, 255));
	CPen    notPen(PS_DASH,  1, RGB(  0,   0,   0));
	CPen    alpPen(PS_DOT,   1, RGB(  0,   0,   0));
	CPen ChosenPen(PS_SOLID, 5, RGB(255, 255, 255));
	POINT myPoint;

	dc.SelectObject(&LimitsPen);
	myPoint.x = m_iXrgb;           myPoint.y = m_iYrgb-m_iYsize;  dc.MoveTo(myPoint);
	myPoint.x = m_iXrgb+m_iXsize;  myPoint.y = m_iYrgb-m_iYsize;  dc.LineTo(myPoint);
	myPoint.x = m_iXrgb;           myPoint.y = m_iYrgb;           dc.MoveTo(myPoint);
	myPoint.x = m_iXrgb+m_iXsize;  myPoint.y = m_iYrgb;           dc.LineTo(myPoint);

	myPoint.x = m_iXalp;           myPoint.y = m_iYalp-m_iYsize;  dc.MoveTo(myPoint);
	myPoint.x = m_iXalp+m_iXsize;  myPoint.y = m_iYalp-m_iYsize;  dc.LineTo(myPoint);
	myPoint.x = m_iXalp;           myPoint.y = m_iYalp;           dc.MoveTo(myPoint);
	myPoint.x = m_iXalp+m_iXsize;  myPoint.y = m_iYalp;           dc.LineTo(myPoint);
//
			m_iKnotIndexRGB;
			m_iKnotIndexALP;
//
	int i, iyscale;
	iyscale = 256 / m_iYsize;
	long iShift;

	if(m_iKnotIndexRGB >= 0) {
		dc.SelectObject(&ChosenPen);
		myPoint.x = m_iXrgb + m_iKnotIndexRGB * 2;
		myPoint.y = m_iYrgb;
		dc.MoveTo(myPoint);
		myPoint.y = m_iYrgb - m_iYsize;
		dc.LineTo(myPoint);
	}
	dc.SelectObject(&redPen);
	for(i=0; i<=m_iNrgbNots; i++) {
		if(m_rgbNots[i].red == m_rgbNots[i].grn) { iShift = 1; } else { iShift = 0; }
		myPoint.x = m_iXrgb + m_rgbNots[i].ind * 2;
		myPoint.y = m_iYrgb - m_rgbNots[i].red / iyscale  +  iShift;
		if(i == 0) { dc.MoveTo(myPoint); }
		else       { dc.LineTo(myPoint); }
	}
	dc.SelectObject(&grnPen);
	for(i=0; i<=m_iNrgbNots; i++) {
		myPoint.x = m_iXrgb + m_rgbNots[i].ind * 2;
		myPoint.y = m_iYrgb - m_rgbNots[i].grn / iyscale;
		if(i == 0) { dc.MoveTo(myPoint); }
		else       { dc.LineTo(myPoint); }
	}
	dc.SelectObject(&bluPen);
	for(i=0; i<=m_iNrgbNots; i++) {
		if(m_rgbNots[i].blu == m_rgbNots[i].grn) { iShift = -1; } else { iShift = 0; }
		myPoint.x = m_iXrgb + m_rgbNots[i].ind * 2;
		myPoint.y = m_iYrgb - m_rgbNots[i].blu / iyscale  +  iShift;
		if(i == 0) { dc.MoveTo(myPoint); }
		else       { dc.LineTo(myPoint); }
	}
	dc.SelectObject(&notPen);
	for(i=0; i<=m_iNrgbNots; i++) {
		myPoint.x = m_iXrgb + m_rgbNots[i].ind * 2;
		myPoint.y = m_iYrgb;
		dc.MoveTo(myPoint);
		myPoint.y = m_iYrgb - m_iYsize;
		dc.LineTo(myPoint);
	}

	if(m_iKnotIndexALP >= 0) {
		dc.SelectObject(&ChosenPen);
		myPoint.x = m_iXalp + m_iKnotIndexALP * 2;
		myPoint.y = m_iYalp;
		dc.MoveTo(myPoint);
		myPoint.y = m_iYalp - m_iYsize;
		dc.LineTo(myPoint);
	}
	dc.SelectObject(&alpPen);
	for(i=0; i<=m_iNalpNots; i++) {
		myPoint.x = m_iXalp + m_alpNots[i].ind * 2;
		myPoint.y = m_iYalp - m_alpNots[i].alp / iyscale;
		if(i == 0) { dc.MoveTo(myPoint); }
		else       { dc.LineTo(myPoint); }
	}
	dc.SelectObject(&notPen);
	for(i=0; i<=m_iNalpNots; i++) {
		myPoint.x = m_iXalp + m_alpNots[i].ind * 2;
		myPoint.y = m_iYalp;
		dc.MoveTo(myPoint);
		myPoint.y = m_iYalp - m_iYsize;
		dc.LineTo(myPoint);
	}


//	char str[100];
//	sprintf(str, "%d", m_rgbNots[0].ind);
//	MessageBox(str);
}

void CEditLUT::OnLButtonDown(UINT nFlags, CPoint point) 
{
	int i;
	long diff, diff0, index;

	if(m_crRGB.PtInRect(point)) {
		index = (point.x - m_crRGB.left) / 2;
		diff0 = 1000;
		if(index == 0  || index == 255) { return; }
		for(i=0; i<=m_iNrgbNots; i++) {
			diff = index - m_rgbNots[i].ind;  if(diff < 0) { diff = -diff; }
			if(diff < diff0) { diff0 = diff; m_iChosenNot = i; }
		}
		if(m_iChosenNot == 0)           { Insert_rgbNotAfter(0); }
		if(m_iChosenNot == m_iNrgbNots) { Insert_rgbNotAfter(m_iNrgbNots-1); }
		m_rgbNots[m_iChosenNot].ind = index;
//		DrawPlots();
		m_csKnotIndex.Format("%d", index);
		m_iKnotIndexRGB = index;
		m_iKnotIndexALP = -1;
		UpdateData(FALSE);
		Invalidate();
	}

	if(m_crALP.PtInRect(point)) {
		index = (point.x - m_crALP.left) / 2;
		diff0 = 1000;
		if(index == 0  || index == 255) { return; }
		for(i=0; i<=m_iNalpNots; i++) {
			diff = index - m_alpNots[i].ind;  if(diff < 0) { diff = -diff; }
			if(diff < diff0) { diff0 = diff; m_iAlpChosenNot = i; }
		}
		if(m_iAlpChosenNot == 0)           { Insert_alpNotAfter(0); }
		if(m_iAlpChosenNot == m_iNalpNots) { Insert_alpNotAfter(m_iNalpNots-1); }
		m_alpNots[m_iAlpChosenNot].ind = index;
//		m_alpNots[m_iAlpChosenNot].alp = (m_crALP.bottom - point.y) * 2;
//		DrawPlots();
		m_csKnotIndex.Format("%d", index);
		m_iKnotIndexRGB = -1;
		m_iKnotIndexALP = index;
		UpdateData(FALSE);
		Invalidate();
	}

	CDialog::OnLButtonDown(nFlags, point);
}

void CEditLUT::OnMouseMove(UINT nFlags, CPoint point) 
{
	long index, value, ind, red, grn, blu, alp;

	if(m_iChosenNot >= 0) {
		index = (point.x - m_crRGB.left) / 2;
		if(0 >= index  ||  index >= 255) { return; }
		if(m_rgbNots[m_iChosenNot-1].ind != index  &&  index != m_rgbNots[m_iChosenNot+1].ind) {
			m_rgbNots[m_iChosenNot].ind = index;
			if(m_rgbNots[m_iChosenNot-1].ind > index) {
				ind = m_rgbNots[m_iChosenNot-1].ind;
				red = m_rgbNots[m_iChosenNot-1].red;
				grn = m_rgbNots[m_iChosenNot-1].grn;
				blu = m_rgbNots[m_iChosenNot-1].blu;
				m_rgbNots[m_iChosenNot-1].ind = m_rgbNots[m_iChosenNot].ind;
				m_rgbNots[m_iChosenNot-1].red = m_rgbNots[m_iChosenNot].red;
				m_rgbNots[m_iChosenNot-1].grn = m_rgbNots[m_iChosenNot].grn;
				m_rgbNots[m_iChosenNot-1].blu = m_rgbNots[m_iChosenNot].blu;
				m_rgbNots[m_iChosenNot].ind = ind;
				m_rgbNots[m_iChosenNot].red = red;
				m_rgbNots[m_iChosenNot].grn = grn;
				m_rgbNots[m_iChosenNot].blu = blu;
				m_iChosenNot--;
			}
			if(m_rgbNots[m_iChosenNot+1].ind < index) {
				ind = m_rgbNots[m_iChosenNot+1].ind;
				red = m_rgbNots[m_iChosenNot+1].red;
				grn = m_rgbNots[m_iChosenNot+1].grn;
				blu = m_rgbNots[m_iChosenNot+1].blu;
				m_rgbNots[m_iChosenNot+1].ind = m_rgbNots[m_iChosenNot].ind;
				m_rgbNots[m_iChosenNot+1].red = m_rgbNots[m_iChosenNot].red;
				m_rgbNots[m_iChosenNot+1].grn = m_rgbNots[m_iChosenNot].grn;
				m_rgbNots[m_iChosenNot+1].blu = m_rgbNots[m_iChosenNot].blu;
				m_rgbNots[m_iChosenNot].ind = ind;
				m_rgbNots[m_iChosenNot].red = red;
				m_rgbNots[m_iChosenNot].grn = grn;
				m_rgbNots[m_iChosenNot].blu = blu;
				m_iChosenNot++;
			}

			m_csKnotIndex.Format("%d", index);
			m_iKnotIndexRGB = index;
			m_iKnotIndexALP = -1;
			UpdateData(FALSE);
			Invalidate();
//			DrawPlots();
		}
	}

	if(m_iAlpChosenNot >= 0) {
		index = (point.x - m_crALP.left) / 2;
		if(0 >= index  ||  index >= 255) { return; }
		if(m_alpNots[m_iAlpChosenNot-1].ind != index  &&  index != m_alpNots[m_iAlpChosenNot+1].ind) {
			m_alpNots[m_iAlpChosenNot].ind = index;
			if(m_alpNots[m_iAlpChosenNot-1].ind > index) {
				ind = m_alpNots[m_iAlpChosenNot-1].ind;
				alp = m_alpNots[m_iAlpChosenNot-1].alp;
				m_alpNots[m_iAlpChosenNot-1].ind = m_alpNots[m_iAlpChosenNot].ind;
				m_alpNots[m_iAlpChosenNot-1].alp = m_alpNots[m_iAlpChosenNot].alp;
				m_alpNots[m_iAlpChosenNot  ].ind = ind;
				m_alpNots[m_iAlpChosenNot  ].alp = alp;
				m_iAlpChosenNot--;
			}
			if(m_alpNots[m_iAlpChosenNot+1].ind < index) {
				ind = m_alpNots[m_iAlpChosenNot+1].ind;
				alp = m_alpNots[m_iAlpChosenNot+1].alp;
				m_alpNots[m_iAlpChosenNot+1].ind = m_alpNots[m_iAlpChosenNot].ind;
				m_alpNots[m_iAlpChosenNot+1].alp = m_alpNots[m_iAlpChosenNot].alp;
				m_alpNots[m_iAlpChosenNot  ].ind = ind;
				m_alpNots[m_iAlpChosenNot  ].alp = alp;
				m_iAlpChosenNot++;
			}
			m_csKnotIndex.Format("%d", index);
			m_iKnotIndexRGB = -1;
			m_iKnotIndexALP = index;
			UpdateData(FALSE);
			Invalidate();
//			DrawPlots();
		}
	}

	if(m_iAlpValCosenNot >= 0) {
		value = (m_crALP.bottom - point.y) * 2;
		if(0 <= value  &&  value < 255) {
			m_alpNots[m_iAlpValCosenNot].alp = value;
//			DrawPlots();
			Invalidate();
		}
	}

	CDialog::OnMouseMove(nFlags, point);
}

void CEditLUT::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if(m_iChosenNot >= 0  ||  m_iAlpChosenNot >= 0) {
		SetLUTirgba();
		RenderIt();
	}

	m_iChosenNot    = -1;
	m_iAlpChosenNot = -1;
	Invalidate();
//	DrawPlots();

	CDialog::OnLButtonUp(nFlags, point);
}

void CEditLUT::Insert_rgbNotAfter(int i0)
{
	int i;
	for(i=m_iNrgbNots; i>i0; i--) {
		m_rgbNots[i+1].ind = m_rgbNots[i].ind;
		m_rgbNots[i+1].red = m_rgbNots[i].red;
		m_rgbNots[i+1].grn = m_rgbNots[i].grn;
		m_rgbNots[i+1].blu = m_rgbNots[i].blu;
	}
	m_iNrgbNots++;
	m_iChosenNot = i0 + 1;
}

void CEditLUT::OnRButtonDown(UINT nFlags, CPoint point) 
{
	int i, i0;
	long diff, diff0, index;
	COLORREF clr;

	if(m_crRGB.PtInRect(point)) {
		index = (point.x - m_crRGB.left) / 2;
		diff0 = 1000;
		for(i=0; i<=m_iNrgbNots; i++) {
			diff = index - m_rgbNots[i].ind;  if(diff < 0) { diff = -diff; }
			if(diff < diff0) { diff0 = diff; i0 = i; }
		}
		m_csKnotIndex.Format("%d", m_rgbNots[i0].ind);
		m_iKnotIndexRGB = m_rgbNots[i0].ind;
		m_iKnotIndexALP = -1;
		UpdateData(FALSE);
		Invalidate();

		clr = RGB( (BYTE)m_rgbNots[i0].red, (BYTE)m_rgbNots[i0].grn, (BYTE)m_rgbNots[i0].blu );
		CColorDialog dlgColors(clr, CC_FULLOPEN);

		if(dlgColors.DoModal() == IDOK) {
			clr = dlgColors.GetColor();
			m_rgbNots[i0].red = (long)GetRValue(clr);
			m_rgbNots[i0].grn = (long)GetGValue(clr);
			m_rgbNots[i0].blu = (long)GetBValue(clr);

			SetLUTirgba();
			RenderIt();
		}
		Invalidate();
//		DrawPlots();
	}

	if(m_crALP.PtInRect(point)) {
		index = (point.x - m_crALP.left) / 2;
		diff0 = 1000;
		for(i=0; i<=m_iNalpNots; i++) {
			diff = index - m_alpNots[i].ind;  if(diff < 0) { diff = -diff; }
			if(diff < diff0) { diff0 = diff; m_iAlpValCosenNot = i; }
		}
		m_csKnotIndex.Format("%d", m_alpNots[m_iAlpValCosenNot].ind);
		m_iKnotIndexRGB = -1;
		m_iKnotIndexALP = m_alpNots[m_iAlpValCosenNot].ind;
		UpdateData(FALSE);
		m_alpNots[m_iAlpValCosenNot].alp = (m_crALP.bottom - point.y) * 2;
		Invalidate();
//		DrawPlots();
	}
}

void CEditLUT::Insert_alpNotAfter(int i0)
{
	int i;
	for(i=m_iNalpNots; i>i0; i--) {
		m_alpNots[i+1].ind = m_alpNots[i].ind;
		m_alpNots[i+1].alp = m_alpNots[i].alp;
	}
	m_iNalpNots++;
	m_iAlpChosenNot = i0 + 1;

}

void CEditLUT::OnRButtonUp(UINT nFlags, CPoint point) 
{
	if(m_iAlpValCosenNot >= 0) {
		SetLUTirgba();
		RenderIt();
	}

	m_iAlpValCosenNot = -1;

	CDialog::OnRButtonUp(nFlags, point);
}


void CEditLUT::DumpLUTcols()
{
	FILE *fp;
	int i0, i1, i2, i3, i, j, red[256], grn[256], blu[256], alp[256];
	float y0, yp;


	for(j=0; j<m_iNrgbNots; j++) {
		i0 = m_rgbNots[j  ].ind;
		i1 = m_rgbNots[j+1].ind;

		yp = (float)(m_rgbNots[j+1].red - m_rgbNots[j].red)/(float)(i1-i0);
		y0 = (float)m_rgbNots[j].red;
		for(i=i0; i<i1; i++) { red[i] = (int)(y0 + yp * (float)(i-i0)); }

		yp = (float)(m_rgbNots[j+1].grn - m_rgbNots[j].grn)/(float)(i1-i0);
		y0 = (float)m_rgbNots[j].grn;
		for(i=i0; i<i1; i++) { grn[i] = (int)(y0 + yp * (float)(i-i0)); }

		yp = (float)(m_rgbNots[j+1].blu - m_rgbNots[j].blu)/(float)(i1-i0);
		y0 = (float)m_rgbNots[j].blu;
		for(i=i0; i<i1; i++) { blu[i] = (int)(y0 + yp * (float)(i-i0)); }
	}
	    i0  = m_rgbNots[m_iNrgbNots].ind;
	red[i0] = m_rgbNots[m_iNrgbNots].red;
	grn[i0] = m_rgbNots[m_iNrgbNots].grn;
	blu[i0] = m_rgbNots[m_iNrgbNots].blu;
	
	fp = fopen("tmp.clut", "w");
	for(i0 =  0; i0 < 64; i0++) {
		i1 = i0 + 64;
		i2 = i1 + 64;
		i3 = i2 + 64;
		fprintf(fp, "%4d %4d %4d %4d  %4d %4d %4d %4d  %4d %4d %4d %4d  %4d %4d %4d %4d\n",
			i0, red[i0], grn[i0], blu[i0],
			i1, red[i1], grn[i1], blu[i1],
			i2, red[i2], grn[i2], blu[i2],
			i3, red[i3], grn[i3], blu[i3] );
	}
	fclose(fp);

	for(j=0; j<m_iNalpNots; j++) {
		i0 = m_alpNots[j  ].ind;
		i1 = m_alpNots[j+1].ind;

		yp = (float)(m_alpNots[j+1].alp - m_alpNots[j].alp)/(float)(i1-i0);
		y0 = (float)m_alpNots[j].alp;
		for(i=i0; i<i1; i++) { alp[i] = (int)(y0 + yp * (float)(i-i0)); }
	}
	    i0  = m_alpNots[m_iNalpNots].ind;
	alp[i0] = m_alpNots[m_iNalpNots].alp;
	
	fp = fopen("tmp.alut", "w");
	for(i0 =  0; i0 < 64; i0++) {
		i1 = i0 + 64;
		i2 = i1 + 64;
		i3 = i2 + 64;
		fprintf(fp, "%4d %4d %4d %4d  %4d %4d %4d %4d  %4d %4d %4d %4d  %4d %4d %4d %4d\n",
			i0, alp[i0], alp[i0], alp[i0],
			i1, alp[i1], alp[i1], alp[i1],
			i2, alp[i2], alp[i2], alp[i2],
			i3, alp[i3], alp[i3], alp[i3] );
	}
	fclose(fp);

}

void CEditLUT::SetHVRport(Ihvr_work *p_hvr_work)
{
	pChvrMainPort[0] = p_hvr_work;
}

void CEditLUT::SetLUTirgba()
{
	HRESULT hr;
	int i, indRGB[32], indAlp[32];
	float fRed[32], fGrn[32], fBlu[32], fAlp[32];

/*****************************************************************
 *	Copy RGBA nots into arrays suitable for hvr_server interface *
 *****************************************************************/
	int iNrgb = m_iNrgbNots + 1;
	int iNalp = m_iNalpNots + 1;
	for(i=0; i<iNalp; i++) {
		indAlp[i] =   (int)m_alpNots[i].ind;
		fAlp[i]   = (float)m_alpNots[i].alp / (float)255.;
	}
	for(i=0; i<iNrgb; i++) {
		indRGB[i] =   (int)m_rgbNots[i].ind;
		fRed[i]   = (float)m_rgbNots[i].red / (float)255.;
		fGrn[i]   = (float)m_rgbNots[i].grn / (float)255.;
		fBlu[i]   = (float)m_rgbNots[i].blu / (float)255.;
	}

/*****************************************************************
 *	Set RGBA LUTs in hvr_server and render with new LUTs         *
 *****************************************************************/
	hr = pChvrMainPort[0]->SetLUTia(iNalp, indAlp, fAlp);
	hr = pChvrMainPort[0]->SetLUTirgb(iNrgb, indRGB, fRed, fGrn, fBlu);
}

void CEditLUT::OnLutButtonSave()
{
	int i;
	FILE *fp;
	CFileDialog dlgFile(FALSE);

	dlgFile.m_ofn.lpstrFilter = "Color RGBA LUT Files (*.lut)\0*.lut\0\0";

	if(dlgFile.DoModal() == IDOK) {	
		CString csFile = dlgFile.GetPathName();
		if((LPCTSTR)csFile != "") {
			fp = fopen((LPCTSTR)csFile, "w");
			fprintf(fp, "NrgbNots = %3d    NalphaNots = %3d\n", m_iNrgbNots, m_iNalpNots);
			for(i = 0; i <= m_iNrgbNots; i++) {
				fprintf(fp, "IndRedGrnBlu    %5d %7d %5d %5d\n",m_rgbNots[i].ind,
																m_rgbNots[i].red,
																m_rgbNots[i].grn,
																m_rgbNots[i].blu  );
			}
			for(i = 0; i <= m_iNalpNots; i++) {
				fprintf(fp, "IndAlpha        %5d %7d\n", m_alpNots[i].ind, m_alpNots[i].alp);
			}
			fclose(fp);
		}
	}
}

void CEditLUT::OnLutButtonLoad() 
{
	CFileDialog dlgFile(TRUE);

	dlgFile.m_ofn.lpstrFilter = "Color RGBA LUT Files (*.lut)\0*.lut\0\0";

	if(dlgFile.DoModal() == IDOK) {	
		CString csFile = dlgFile.GetPathName();
		if((LPCTSTR)csFile != "") {
		ReadLUTfile((LPCTSTR)csFile);
			m_csKnotIndex.Format(" ");
			m_iKnotIndexRGB = -1;
			m_iKnotIndexALP = -1;
			UpdateData(FALSE);
			Invalidate();
//			DrawPlots();
			SetLUTirgba();
			RenderIt();
		}
	}

}

void CEditLUT::ReadLUTfile(const char *cFile)
{
	int i;
	FILE *fp;
	char str[256];
	int match( char *, char *, int );

	if( match( (char *)cFile, "GrayRamp",8) ) {
		SetLUTtoGrayScale();
	} else {
		fp = fopen(cFile, "r");
		if(fp != NULL) {
			fscanf(fp, "NrgbNots = %3d    NalphaNots = %3d\n", &m_iNrgbNots, &m_iNalpNots);
			for(i = 0; i <= m_iNrgbNots; i++) {
				fscanf(fp, "IndRedGrnBlu    %5d %7d %5d %5d\n", &m_rgbNots[i].ind,
																&m_rgbNots[i].red,
																&m_rgbNots[i].grn,
																&m_rgbNots[i].blu  );
			}
			for(i = 0; i <= m_iNalpNots; i++) {
				fscanf(fp, "IndAlpha        %5d %7d\n", &m_alpNots[i].ind, &m_alpNots[i].alp);
			}
			fclose(fp);
		} else {
			sprintf(str, "Could NOT open LUT file %s : defaulting to Gray Scale ramp.", cFile);
			MessageBox(str);
			SetLUTtoGrayScale();
		}
	}
}

int CEditLUT::SetLUTfromHVRwork(SLutNots Slut)
{
	int i;

	m_iNalpNots = Slut.iNAnots - 1;
	for(i=0; i <= m_iNalpNots; i++) {
		m_alpNots[i].ind = Slut.iAind[i];
		m_alpNots[i].alp = (long)((float)255.0 * Slut.fAlp[i]);
	}

	m_iNrgbNots = Slut.iNCnots - 1;
	for(i=0; i <= m_iNrgbNots; i++) {
		m_rgbNots[i].ind = (long) Slut.iCind[i];
		m_rgbNots[i].red = (long)((float)255.0 * Slut.fRed[i]);
		m_rgbNots[i].grn = (long)((float)255.0 * Slut.fGrn[i]);
		m_rgbNots[i].blu = (long)((float)255.0 * Slut.fBlu[i]);
	}
	
	return 0;
}

void CEditLUT::OnShowWindow(BOOL bShow, UINT nStatus) 
{
	CDialog::OnShowWindow(bShow, nStatus);
}

void CEditLUT::SetLUTtoGrayScale()
{
		m_iNrgbNots = 1;
		m_iNalpNots = 1;
		m_rgbNots[0].ind =   0;
		m_rgbNots[0].red =   0;
		m_rgbNots[0].grn =   0;
		m_rgbNots[0].blu =   0;
		m_rgbNots[1].ind = 255;
		m_rgbNots[1].red = 255;
		m_rgbNots[1].grn = 255;
		m_rgbNots[1].blu = 255;
		m_alpNots[0].ind =   0;
		m_alpNots[0].alp =   0;
		m_alpNots[1].ind = 255;
		m_alpNots[1].alp = 255;
}

void CEditLUT::OnPaint() 
{
//	CPaintDC dc(this); // device context for painting
	
	DrawPlots();
	
	// Do not call CDialog::OnPaint() for painting messages
}

void CEditLUT::OnLutDeleteKnot() 
{
	int i, i0;

	i0 = -1;
	if(m_iKnotIndexRGB >= 0) {
		if(m_iKnotIndexRGB == 0  ||  m_iKnotIndexRGB == 255) {
			MessageBox("Can not delete an endpoint knot!");	
			return;
		}
		for(i=0; i<=m_iNrgbNots; i++) {
			if(m_iKnotIndexRGB == m_rgbNots[i].ind) { i0 = i; }
		}
		if(i0 > 0) {
			for(i=i0; i<m_iNrgbNots; i++) {
				m_rgbNots[i].ind = m_rgbNots[i+1].ind;
				m_rgbNots[i].red = m_rgbNots[i+1].red;
				m_rgbNots[i].grn = m_rgbNots[i+1].grn;
				m_rgbNots[i].blu = m_rgbNots[i+1].blu;
			}
			m_iNrgbNots--;
			m_iKnotIndexRGB = -1;
			m_csKnotIndex.Format(" ");

			SetLUTirgba();
			RenderIt();
			UpdateData(FALSE);
			Invalidate();
		} else {
			MessageBox("Could NOT find alpha knot for this index!");
		}
	}
	if(m_iKnotIndexALP >= 0) {
		if(m_iKnotIndexALP == 0  ||  m_iKnotIndexALP == 255) {
			MessageBox("Can not delete an endpoint knot!");
			return;
		}
		for(i=0; i<=m_iNalpNots; i++) {
			if(m_iKnotIndexALP == m_alpNots[i].ind) { i0 = i; }
		}
		if(i0 > 0) {
			for(i=i0; i<m_iNalpNots; i++) {
				m_alpNots[i].ind = m_alpNots[i+1].ind;
				m_alpNots[i].alp = m_alpNots[i+1].alp;
			}
			m_iNalpNots--;
			m_iKnotIndexALP = -1;
			m_csKnotIndex.Format(" ");

			SetLUTirgba();
			RenderIt();
			UpdateData(FALSE);
			Invalidate();
		} else {
			MessageBox("Could NOT find alpha knot for this index!");
		}
	}
}

void CEditLUT::SetPP(CHvr_mfcDlg *pp)
{
	pMainDlg = pp;
}

void CEditLUT::RenderIt()
{
	// 1st, stop any QH renderings and wait for Rendering Quality parameters to be reset
	if(pMainDlg->m_iAmRendering == 1) {
		pMainDlg->m_iStop = 1;
		pChvrMainPort[0]->StopRendering();
		while(pMainDlg->m_iAmRendering == 1) { _sleep(10); }
	}
	HRESULT hr = pChvrMainPort[0]->SetDoRender();
}
