// DisplayControl.cpp : implementation file
//
////////////////////////////////////////////////////////////////////////////////
//
// Author:   David H. Porter
//           Laboratory for Computational Science & Engineering
//           University of Minnesota
//
////////////////////////////////////////////////////////////////////////////////
/***************************************************************************************
Program:  HVR_SERVER  --  Hierarchical Volume Rendering Component Object Moduel
Copyright (C) 2002  David H. Porter

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************************/

#include "stdafx.h"
#include "hvr_mfc.h"
#include "DisplayControl.h"
#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CDisplayControl dialog


CDisplayControl::CDisplayControl(CWnd* pParent /*=NULL*/)
	: CDialog(CDisplayControl::IDD, pParent)
{
	//{{AFX_DATA_INIT(CDisplayControl)
	m_iDisplayWidth = 0;
	m_iDisplayHeight = 0;
	m_fDisplayFOV = 0.0f;
	m_iPanelWidth = 0;
	m_iPanelHeight = 0;
	m_iNPanelHoriz = 0;
	m_iNPanelVert = 0;
	m_fFullFOV = 0.0f;
	m_fSeamAngle = 0.0f;
	m_fStereoEyeDisp = 0.0f;
	//}}AFX_DATA_INIT
}


void CDisplayControl::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CDisplayControl)
	DDX_Text(pDX, IDC_DC_EDIT_WIDTH, m_iDisplayWidth);
	DDV_MinMaxInt(pDX, m_iDisplayWidth, 1, 2048);
	DDX_Text(pDX, IDC_DC_EDIT_HEIGHT, m_iDisplayHeight);
	DDV_MinMaxInt(pDX, m_iDisplayHeight, 1, 2048);
	DDX_Text(pDX, IDC_DC_EDIT_FOV, m_fDisplayFOV);
	DDV_MinMaxFloat(pDX, m_fDisplayFOV, 0.f, 170.f);
	DDX_Text(pDX, IDC_EDIT_PAN_WIDTH, m_iPanelWidth);
	DDV_MinMaxInt(pDX, m_iPanelWidth, 1, 2048);
	DDX_Text(pDX, IDC_EDIT_PAN_HEIGHT, m_iPanelHeight);
	DDV_MinMaxInt(pDX, m_iPanelHeight, 1, 2048);
	DDX_Text(pDX, IDC_EDIT_N_PAN_HORIZ, m_iNPanelHoriz);
	DDV_MinMaxInt(pDX, m_iNPanelHoriz, 1, 100);
	DDX_Text(pDX, IDC_EDIT_N_PAN_VERT, m_iNPanelVert);
	DDX_Text(pDX, IDC_EDIT_FULL_FOV, m_fFullFOV);
	DDV_MinMaxFloat(pDX, m_fFullFOV, 0.f, 179.f);
	DDX_Text(pDX, IDC_EDIT_SEAM_ANGLE, m_fSeamAngle);
	DDV_MinMaxFloat(pDX, m_fSeamAngle, -360.f, 360.f);
	DDX_Text(pDX, IDC_EDIT_STEREO_DISPLACEMENT, m_fStereoEyeDisp);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CDisplayControl, CDialog)
	//{{AFX_MSG_MAP(CDisplayControl)
	ON_BN_CLICKED(IDC_BUTTON_AUTO_SET_INTERACTIVE, OnButtonAutoSetInteractive)
	ON_BN_CLICKED(IDC_DC_BUTTON_SAVE, OnDcButtonSave)
	ON_BN_CLICKED(IDC_BUTTON_LOAD, OnButtonLoad)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CDisplayControl message handlers

void CDisplayControl::OnOK() 
{
	UpdateData(TRUE);

	GenerateMovieFrusta();

	CDialog::OnOK();
}

void CDisplayControl::OnButtonAutoSetInteractive() 
{
	int iFullWidth, iFullHeight;
	float fAspectRatio;

	UpdateData(TRUE);

	m_fDisplayFOV = m_fFullFOV;

	// Brutal Hack: assume flat wall for generating "Equivalent Interactive view"
	// Note, to do better, for the general case, must be able to render behind the
	// viewer's head.
	iFullWidth  = m_iNPanelHoriz * m_iPanelWidth;
	iFullHeight = m_iNPanelVert  * m_iPanelHeight;
	if(iFullWidth <= 768  &&  iFullHeight <= 768) {
		m_iDisplayWidth  = iFullWidth;
		m_iDisplayHeight = iFullHeight;
	} else {
		fAspectRatio = (float)iFullHeight / (float)iFullWidth;
		if(fAspectRatio < 1.) {
			m_iDisplayWidth  = 768;
			m_iDisplayHeight = (int)(fAspectRatio * (float)m_iDisplayWidth);
		} else {
			m_iDisplayHeight = 768;
			m_iDisplayWidth  = (int)((float)m_iDisplayHeight / fAspectRatio);
		}
	}
	if(m_fSeamAngle > 0.1) {
		MessageBox("WARNING: seam angle is nonzero: wall and interactive views will differ");
	}

	UpdateData(FALSE);
}

void CDisplayControl::GenerateMovieFrusta()
{
	float fHalfHeight, fPanWidth, fPanHeight, fHalfWidth, fRadius, fAngle;
	float Pid180, half, one, zero;
	float fPanCentX[100], fPanCentY[100], fPanCentZ[100];
	float fPanNormX[100], fPanNormY[100], fPanNormZ[100];
	int ix, iz;

	half   = (float)0.5;
	one    = (float)1.0;
	zero   = (float)0.0;
	Pid180 = (float)3.141592654 / (float)180.;
	fHalfHeight = (float)tan( Pid180 * m_fFullFOV * half );   // Assumes panel is at unit dist. from eye

	fPanHeight = (float)2.0 * fHalfHeight / (float)m_iNPanelVert;

	fPanWidth = fPanHeight * (float)m_iPanelWidth / (float)m_iPanelHeight;

	fHalfWidth = fHalfHeight * (float)m_iNPanelHoriz / (float)m_iNPanelVert;

	for(iz=0; iz < m_iNPanelVert; iz++) {
		fPanCentZ[iz] = fPanHeight * ((float)iz - half * ((float)m_iNPanelVert - 1));
		fPanNormZ[iz] = zero;
	}

	if(m_fSeamAngle < 0.00001) {

		// Flat wall case
		for(ix=0; ix<m_iNPanelHoriz; ix++) {
			fPanCentX[ix] = fPanWidth * ((float)ix - half * ((float)m_iNPanelHoriz - 1));
			fPanCentY[ix] = one;
			fPanNormX[ix] = zero;
			fPanNormY[ix] = one;
		}
	} else {

		//Curved wall case:
		fRadius = (float)0.5 * fPanWidth / (float)tan( Pid180 * m_fSeamAngle * (float)0.5 );
		for(ix=0; ix<m_iNPanelHoriz; ix++) {
			fAngle  = Pid180 * m_fSeamAngle * ((float)ix - half * (float)(m_iNPanelHoriz - 1));
			fPanCentX[ix] = fRadius * (float)sin(fAngle);
			fPanCentY[ix] = fRadius * (float)cos(fAngle) - fRadius + one;
			fPanNormX[ix] = (float)sin(fAngle);
			fPanNormY[ix] = (float)cos(fAngle);
		}
	}

	m_iNMovieFrusta = 0;
	for(iz=0; iz<m_iNPanelVert;  iz++) {
	for(ix=0; ix<m_iNPanelHoriz; ix++) {

		m_SMovieFrusta[m_iNMovieFrusta].iHeight  = m_iPanelHeight;
		m_SMovieFrusta[m_iNMovieFrusta].iWidth   = m_iPanelWidth;
		m_SMovieFrusta[m_iNMovieFrusta].fWidth   = fPanWidth;
		m_SMovieFrusta[m_iNMovieFrusta].fCent[0] = fPanCentX[ix];
		m_SMovieFrusta[m_iNMovieFrusta].fCent[1] = fPanCentY[ix];
		m_SMovieFrusta[m_iNMovieFrusta].fCent[2] = fPanCentZ[iz];
		m_SMovieFrusta[m_iNMovieFrusta].fNorm[0] = fPanNormX[ix];
		m_SMovieFrusta[m_iNMovieFrusta].fNorm[1] = fPanNormY[ix];
		m_SMovieFrusta[m_iNMovieFrusta].fNorm[2] = fPanNormZ[iz];
		m_SMovieFrusta[m_iNMovieFrusta].fUp[0]   = zero;
		m_SMovieFrusta[m_iNMovieFrusta].fUp[1]   = zero;
		m_SMovieFrusta[m_iNMovieFrusta].fUp[2]   = one;

		m_iNMovieFrusta++;
	}}

}

void CDisplayControl::OnDcButtonSave() 
{
	CFileDialog myFileDialog(FALSE);
	FILE *fp;
	int i;
	float fPanHalfWidth, fPanHalfHeight;
	float xx[3], fHoriz[3], fHW, fHH;
	int j;

	UpdateData(TRUE);

	myFileDialog.m_ofn.lpstrFilter = "Display Parameter Files (*.dpf)\0*.dpf\0\0";

	if(myFileDialog.DoModal() == IDOK) {	
		CString csFile = myFileDialog.GetPathName();
		if((LPCTSTR)csFile != "") {

			GenerateMovieFrusta();

			fp = fopen((LPCTSTR)csFile, "w");
			fprintf(fp, "%6d %6d %f     %4d %4d %f %f\n", m_iPanelWidth, m_iPanelHeight, m_fFullFOV,
														  m_iNPanelHoriz, m_iNPanelVert,
														  m_fSeamAngle);

			fprintf(fp, "%6d %6d %f\n", m_iDisplayWidth, m_iDisplayHeight, m_fDisplayFOV, m_fStereoEyeDisp);
			fprintf(fp, "Movie   : width, height, FullFOV,      Nhoriz, Nvert, SeamAng\n");
			fprintf(fp, "Display : width, height, FOV, Eye Displacment\n\n");

			fprintf(fp, "Number of Panels : %d\n", m_iNMovieFrusta);
			fprintf(fp, "PW_CONFIGURATION file:\n");
			fprintf(fp, "NormX,   NormY,   NormZ,     UpX,     UpY,     UpZ,       CentX,   CentY,   CentZ,  HalfWidth HalfHeight\n");
			for(i=0; i<m_iNMovieFrusta; i++) {

				fPanHalfWidth  = (float)0.5 * m_SMovieFrusta[i].fWidth;
				fPanHalfHeight = fPanHalfWidth * (float)m_SMovieFrusta[i].iHeight /
												 (float)m_SMovieFrusta[i].iWidth;
				fprintf(fp, "%f %f %f   %f %f %f   %f %f %f   %f %f\n",
					m_SMovieFrusta[i].fNorm[0],
					m_SMovieFrusta[i].fNorm[1],
					m_SMovieFrusta[i].fNorm[2],
					m_SMovieFrusta[i].fUp[0],
					m_SMovieFrusta[i].fUp[1],
					m_SMovieFrusta[i].fUp[2],
					m_SMovieFrusta[i].fCent[0],
					m_SMovieFrusta[i].fCent[1],
					m_SMovieFrusta[i].fCent[2],
					fPanHalfWidth,
					fPanHalfHeight               );

			}

			fprintf(fp, "\n\n  Panel Verticies:\n");
			for(i=0; i<m_iNMovieFrusta; i++) {
				fprintf(fp, "Panel No %2d\n", i);

				fHW = (float)0.5 * m_SMovieFrusta[i].fWidth;
				fHH = fHW * (float)m_SMovieFrusta[i].iHeight /
							(float)m_SMovieFrusta[i].iWidth;

				fHoriz[0] = m_SMovieFrusta[i].fNorm[1]*m_SMovieFrusta[i].fUp[2]
						  - m_SMovieFrusta[i].fNorm[2]*m_SMovieFrusta[i].fUp[1];

				fHoriz[1] = m_SMovieFrusta[i].fNorm[2]*m_SMovieFrusta[i].fUp[0]
						  - m_SMovieFrusta[i].fNorm[0]*m_SMovieFrusta[i].fUp[2];

				fHoriz[2] = m_SMovieFrusta[i].fNorm[0]*m_SMovieFrusta[i].fUp[1]
						  - m_SMovieFrusta[i].fNorm[1]*m_SMovieFrusta[i].fUp[0];

				for(j=0; j<3; j++) {
					xx[j] = m_SMovieFrusta[i].fCent[j] - fHW * fHoriz[j]
													   - fHH * m_SMovieFrusta[i].fUp[j];
				}
				fprintf(fp, "xyz= %f %f %f\n", xx[0], xx[1], xx[2]);

				for(j=0; j<3; j++) {
					xx[j] = m_SMovieFrusta[i].fCent[j] - fHW * fHoriz[j]
													   + fHH * m_SMovieFrusta[i].fUp[j];
				}
				fprintf(fp, "xyz= %f %f %f\n", xx[0], xx[1], xx[2]);

				for(j=0; j<3; j++) {
					xx[j] = m_SMovieFrusta[i].fCent[j] + fHW * fHoriz[j]
													   - fHH * m_SMovieFrusta[i].fUp[j];
				}
				fprintf(fp, "xyz= %f %f %f\n", xx[0], xx[1], xx[2]);

				for(j=0; j<3; j++) {
					xx[j] = m_SMovieFrusta[i].fCent[j] + fHW * fHoriz[j]
													   + fHH * m_SMovieFrusta[i].fUp[j];
				}
				fprintf(fp, "xyz= %f %f %f\n", xx[0], xx[1], xx[2]);

			}

			fclose(fp);
		}
	}
}

void CDisplayControl::OnButtonLoad() 
{
	CFileDialog myFileDialog(TRUE);
	FILE *fp;

	myFileDialog.m_ofn.lpstrFilter = "Display Parameter Files (*.dpf)\0*.dpf\0\0";

	if(myFileDialog.DoModal() == IDOK) {	
		CString csFile = myFileDialog.GetPathName();
		if((LPCTSTR)csFile != "") {

			fp = fopen((LPCTSTR)csFile, "r");
			fscanf(fp, "%6d %6d %f     %4d %4d %f\n", &m_iPanelWidth, &m_iPanelHeight, &m_fFullFOV,
													  &m_iNPanelHoriz, &m_iNPanelVert,
													  &m_fSeamAngle);

			fscanf(fp, "%6d %6d %f %f\n", &m_iDisplayWidth, &m_iDisplayHeight, &m_fDisplayFOV, &m_fStereoEyeDisp);

			fclose(fp);

			UpdateData(FALSE);
		}
	}
}
