program GenBlockTree
!
! Fortran 90 Routine: GenBlockTree
! This is an example of how to generate a block tree hierarchy
! from data on a uniform grid by calling subroutine block_tree.
! For very large grids, data may be staged in "slabs" which span
! the full X and Y grid, but span only a subrange of Z.
!
! Two examples are provided:
! 1) full data does not fit in memory, search for
!         Use this if your grid does NOT fit in memory
! 2) full data fits in memory, search for
!         Use this if your grid DOES fit in memory
! The 2nd example is commented out here.
! The Z-thickness of the slab used by subroutine block_tree (i.e., nzslab)
! shuold be 16 in either case.
! 
! Author:  David H. Porter
!          LCSE, University of Minnesota
! Date:    20 March 2002
!

! Declare are variables and arrays; allocate the small ones.
      implicit none
      real, allocatable, dimension(:,:,:) :: field
      real flimits(6)
      integer*1, allocatable, dimension(:) :: ucData
      integer :: i, j, k, nx, ny, nz, iBlock, nzoff, nzslab, ijk
      integer :: ivalue
      integer :: fulldim(3), curdim(3), curoff(3)
      real :: scale, offset
      character (len=10) :: hvfile

! Set geometrical limits of domain to be rendered
      flimits(1) = -1.0
      flimits(2) =  1.0
      flimits(3) = -1.0
      flimits(4) =  1.0
      flimits(5) = -1.0
      flimits(6) =  1.0

! Set full dimensions of grid
      nx = 256          ! YOUR GRID DIMENSIONS GO HERE
      ny = 256          ! YOUR GRID DIMENSIONS GO HERE
      nz = 256          ! YOUR GRID DIMENSIONS GO HERE
      nzslab = 16       ! should be a power of 2 and divide evenly into nz

! Allocate big arrays
      allocate(ucData(nx*ny*nzslab))
      allocate(field(nx,ny,nzslab))   ! Use this if grid does NOT fit in memory
!      allocate(field(nx,ny,nz))      ! Use this if grid DOES fit in memory

! Set scaling parameters for map from fval (float) to ival (1-byte int) via
!              ival = int(offset + scale * fval)
! values of ival are clipped to the range [0, 255]
! Choose offset and scale to map typical [Min, Max] values to [0, 255]
      offset = 128.0         ! YOUR OFFSET GOES HERE
      scale  = 128.0         ! YOUR SCALING FACTOR GOES HERE

! Set name of output file (should end in .hv)
! WARNING: character array hvfile must be exaclty the length of the string
!          it contains.  For example, Test256.hv is 10 characters long and
!          array hvfile is declaired to be 10 charters above.
      write (hvfile, '("Test256.hv")')
      write (*, '(A)') hvfile

! Set parameters for generation of Block Tree
      iBlock = 126      ! should be 2**n - 2, where n is a positive integer
      fulldim(1) = nx
      fulldim(2) = ny
      fulldim(3) = nz
    
! Fill in field array (use this if grid DOES fit in memory)
!      do k = 1, nz
!      do j = 1, ny
!      do i = 1, nx
!                       ! YOUR DATA GOES HERE (if it fits in memory)
!        field(i,j,k) = sin( 20.0 * float(i+j+k) / float(nx) )
!      enddo
!      enddo
!      enddo

! Loop over slabs
      do nzoff = 1, nz, nzslab

! Fill in field array (use this if grid does NOT fit in memory)
        do k = nzoff, nzoff+nzslab-1
        do j = 1, ny
        do i = 1, nx
                                 ! YOUR DATA GOES HERE (if it does not fit)
          field(i,j,1+k-nzoff) = sin( 20.0 * float(i+j+k) / float(nx) )
        enddo
        enddo
        enddo

! Set offset and size arrays for this slab
        curoff(1) = 0
        curoff(2) = 0
        curoff(3) = nzoff - 1    ! C subroutine starts counting from 0
        curdim(1) = nx
        curdim(2) = ny
        curdim(3) = min(nzslab, 1+nz-nzoff)
          
! Map field data to [0, 255] and copy into 1-byte array
        ijk = 1
        do k = nzoff, nzoff+nzslab-1
        do j = 1, ny
        do i = 1, nx

          ! Use this if your grid does NOT fit in memory
          ivalue = int(offset + scale * field(i, j, 1+k-nzoff)) 

           ! Use this if your grid DOES fit in memory
!          ivalue = int(offset + scale * field(i, j, k)) 

          if(ivalue .gt. 255) ivalue = 255
          if(ivalue .lt.   0) ivalue =   0
          ucData(ijk) = ivalue
          ijk = ijk + 1
        enddo
        enddo
        enddo

! Call BlockTree for this slab
        call block_tree(hvfile,fulldim,curdim,curoff,iBlock,flimits,ucData)
  enddo

end program GenBlockTree
