/* Subroutine block_tree
 * This routine produces a block tree hierarchy from
 * data on a uniform grid.
 *
 * Arguments
 * char *pfile           Name of file into block tree data is written
 * int fulldim[3]        XYZ dimensions of full uniform grid
 * int curdim[3]         XYZ dimensions of current input sub-slab of full grid
 * int curoff[3]         XYZ offset of input sub-slab into full grid
 * int *pblock           Pointer to interior size of each block
 * unsigned char *field  Pointer to array containing input slab
 *
 * In order to facilitate efficient and seamless trilinear interpolation,
 * routine Block_Tree writes blocks with overlapping boundaries which are
 * 1 voxel deep.  2D and 3D textures are best supported if thier full
 * dimensions -- including boundaries -- are powers of 2.  Hence, the interior
 * dimensions of each block (*pblock) should be of the form 2**n-2, where
 * n is a positive integer.  Block sizes of serveral MB are fairly optimal for
 * both disk IO speed and memory managment.  Hence a good value for *pblock
 * is 126.
 *
 * Author:  David H. Porter
 *          LCSE, University of Minnesota
 * Date:    25 April 2003
 * All rights reserved
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <math.h>

#define MAX_NODES 8192
#define MAX_CHILDREN 64
#define NBLACK 4*1024*1024
#define FILE_BLOCK 64*1024

static unsigned char *ucSlab;
static unsigned char *fieldav[65];
static unsigned char *fieldmin[65];
static unsigned char *fieldmax[65];

void block_tree_(char *pfile, int fulldim[3], int curdim[3], int curoff[3],
                 int *pblock, float fLimits[6], unsigned char *field)
{
	int nx, ny, nz, nx1, ny1, nz1, iBlock;
	static int midpoint[MAX_NODES][3];
	static off64_t byteoff, tile_byteoff[MAX_NODES];
	static int tile_dim[MAX_NODES][3], tile_offset[MAX_NODES][3], tile_rep[MAX_NODES];
	static int tile_parent[MAX_NODES], tile_error[MAX_NODES];
	static int tile_children[MAX_NODES][MAX_CHILDREN], tile_N_children[MAX_NODES];
	static int irep, ireplast, nlines, iNnodes=0;
	int i, j, k, ih, jh, kh, node, node_root, ichild, more_nodes, maxerrtmp;
	int i0, j0, k0, i1, j1, k1;
	off64_t nblocks, leftover;
	static int fd;
	static int iprint = 0;
	int irl, jrl, krl, irh, jrh, krh;
	int itl, jtl, ktl, ith, jth, kth;
	int iil, jil, kil, iih, jih, kih;
	int irn, jrn, krn, irl0, jrl0, krl0, irh0, jrh0, krh0, iOff, nChar;
	int ks, kr, js, ir, jr, ijks, ijkr, nslab, jksoff, jkroff;
	int itn, jtn, ktn, jtoff, jroff, it, iVal, iMin, iMax, iErr;
	float fPmin, fPmax, repfac, fVal;
	int vmin, vmax, v0, v1, v2, v3, v4, v5, v6, v7, iadd, jadd, kadd, ijk0, ijk1;
	int nx0, ny0, nz0, nx2, ny2, nz2, nxy0, i2, j2, k2, irep1, irep0, jkoff, jkoff2;
	int ijkd, itlu0, ithu0, itlu, ithu, jtlu, jthu, ktlu, kthu, is, kilu, kihu;
	static unsigned char fSlab[NBLACK],  fSlabMin[NBLACK],  fSlabMax[NBLACK];
	static int iDoRep[65];
	int iIntersection, nxyzmax, iBndry;
	off64_t i64Offset;
	char cLine[256];
	long dwSize;
	void TreeScanOffsets(int node, off64_t *pbyteoff, off64_t tile_byteoff[MAX_NODES],
						 int irep, int tile_rep[MAX_NODES], int tile_dim[MAX_NODES][3],
						 int tile_N_children[MAX_NODES], int tile_children[MAX_NODES][MAX_CHILDREN]);

	// Dimensions of full domain, and max there of
	nx = fulldim[0];
	ny = fulldim[1];
	nz = fulldim[2];
	nxyzmax = nx;
	if(ny > nxyzmax) { nxyzmax = ny; }
	if(nz > nxyzmax) { nxyzmax = nz; }
	iBlock = *pblock;

	// Generate tile decomposition layout on 1st pass
	if(iNnodes == 0) {
		nlines     = 2 * (1+nx/iBlock)*(1+ny/iBlock)*(1+nz/iBlock);           // Max # of lines for hv info
		byteoff = (off64_t)(FILE_BLOCK * (1 + (80 * nlines) / (FILE_BLOCK)));   // offset to start of byte info

		for(irep=0; irep<65; irep++) { iDoRep[irep] = 0; }
		node = 0;
		irep = 1;
		more_nodes = 2;
		while(more_nodes > 1) {
			iDoRep[irep] = 1;

			iBlock = *pblock;
//			if(nxyzmax > irep*(iBlock+2)  ||  nxyzmax <= irep*iBlock)
			if(nxyzmax > irep*(iBlock+2)) {
				iBndry = 1;		// Here, tiles need overlapping boundaries or it fits in one tile even w/ bndrys
			} else {
				iBndry = 0;		// Here, tile spans entire domain, no overlapping boundaries needed
				iBlock += 2;
			}


			more_nodes = 0;
			for(k=0; k<nz; k += iBlock*irep) { kh = k+iBlock*irep;  if(kh > nz) { kh = nz; }
			for(j=0; j<ny; j += iBlock*irep) { jh = j+iBlock*irep;  if(jh > ny) { jh = ny; }
			for(i=0; i<nx; i += iBlock*irep) { ih = i+iBlock*irep;  if(ih > nx) { ih = nx; }
				// midpoint of node will be used in determining parent node
				midpoint[node][0] = (i+ih) / 2;
				midpoint[node][1] = (j+jh) / 2;
				midpoint[node][2] = (k+kh) / 2;

				// xyz size of tile including 2 boundary zones CHANGE
				tile_dim[node][0] = 2*iBndry + (ih - i + irep - 1) / irep;
				tile_dim[node][1] = 2*iBndry + (jh - j + irep - 1) / irep;
				tile_dim[node][2] = 2*iBndry + (kh - k + irep - 1) / irep;  // if any of these are 2, will skip node!

				// offset (in bytes) from beginning of file to start of tile data
				tile_byteoff[node] = byteoff;
				byteoff += (off64_t)(FILE_BLOCK*(1+
					(tile_dim[node][0]*tile_dim[node][1]*tile_dim[node][2] - 1)
					/(FILE_BLOCK)));

				// replication factor of tile
				tile_rep[node] = irep;

				// xyz offset of tile (including boundary indentation) in units of its own zones
				tile_offset[node][0] = i/irep - iBndry;
				tile_offset[node][1] = j/irep - iBndry;
				tile_offset[node][2] = k/irep - iBndry;

				// Default parent node is -1 (no parent).
				// Set parents of all child nodes based on midpoint of child being inside current node
				// As parents are found for children, tell the parents about it as well.
				tile_parent[node] = -1;
				tile_N_children[node] = 0;
				for(ichild=0; ichild<node; ichild++) {
					if(2*tile_rep[ichild] == irep) {
						if(i < midpoint[ichild][0] && midpoint[ichild][0] < ih &&
						   j < midpoint[ichild][1] && midpoint[ichild][1] < jh &&
						   k < midpoint[ichild][2] && midpoint[ichild][2] < kh   )
						{
							tile_parent[ichild] = node;
							tile_children[node][tile_N_children[node]] = ichild;
							tile_N_children[node]++;
						}
					}
				}

				// Initialize max error in using this tile
				tile_error[node]  =  0;

				// By construction, root node is always the last one in the list
				node_root = node;

				node++;
				more_nodes++;
			}}}
			irep *= 2;
		}
		iNnodes = node;
		node_root = node-1;			// By construction, root node is always the last one in the list

		// Now reset byte offsets into file to be in the order of a tree scan for each level
		// Will leave the root of the tree as the last brick, so that if you see that, then you have it all.
		nlines     = iNnodes + 100;                                           // # of lines for hv info + other header info
		byteoff = (off64_t)(FILE_BLOCK * (1 + (80 * nlines) / (FILE_BLOCK)));   // offset to start of byte info

		// For each replication factor used, do a  tree scan to
		//   1) find nodes at this replication factor, and
		//   2) assign byte offsets into the hv file in tree scan order
		for(irep=1; irep<65; irep++) { if(iDoRep[irep] > 0) {
			TreeScanOffsets(node_root, &byteoff, tile_byteoff,
							irep, tile_rep, tile_dim,
							tile_N_children, tile_children    );
		}}

		// Allocate space for reduced fields
		fieldav[1] = field;
		irep = 2;
//		printf("nx = %d   ny = %d   nz = %d\n", nx, ny, nz);
		while(irep < 65) {
			if(iDoRep[irep] > 0) {
				nx1 = (curdim[0] + irep - 1) / irep;
				ny1 = (curdim[1] + irep - 1) / irep;
				nz1 = (curdim[2] + irep - 1) / irep;
				dwSize = (long)nx1 * (long)ny1 * (long)nz1;
                		fieldav[irep]  = (unsigned char *)valloc((size_t)dwSize);
                		fieldmax[irep] = (unsigned char *)valloc((size_t)dwSize);
                		fieldmin[irep] = (unsigned char *)valloc((size_t)dwSize);
//				printf("field[av,max,min]: irep = %2d, allocated %d bytes each\n", irep, dwSize);
			}
			irep *= 2;
		}

		// Write out file filled with 0s
//		printf("Start write of 0s\n");
      		ucSlab = (unsigned char *)valloc((size_t)NBLACK);
		for(i=0; i<NBLACK; i++) { ucSlab[i] = (unsigned char)0; }
		leftover = byteoff;
		nblocks = 0;
		while(leftover >= (off64_t)NBLACK) { nblocks++; leftover -= (off64_t)NBLACK; }
//		printf("NBLACK=%d   nblocks=%lld   byteoff=%lld   leftover=%lld\n", NBLACK, nblocks, byteoff, leftover);
		if(nblocks > 10000) { exit(0); }
		fd = open(pfile, 0666);
		for(i=0; i<nblocks; i++) { write(fd, ucSlab, NBLACK); }
		if(leftover > 0)         { write(fd, ucSlab, leftover); }
//		printf("Finish write of 0s\n");

	}

	// For every subregion, start by generating all of the reduced resolution fields
//	printf("start averages\n");
	irep1 = 2;
	while(irep1 < 65) {
		if(iDoRep[irep1] > 0) {
			irep0 = irep1 / 2;
			nx1 = (curdim[0] + irep1 - 1) / irep1;
			ny1 = (curdim[1] + irep1 - 1) / irep1;
			nz1 = (curdim[2] + irep1 - 1) / irep1;
			nx0 = (curdim[0] + irep0 - 1) / irep0;
			ny0 = (curdim[1] + irep0 - 1) / irep0;
			nz0 = (curdim[2] + irep0 - 1) / irep0;

			nxy0 = nx0 * ny0;
			for(k1=0; k1<nz1; k1++) { k0 = k1 * 2;
			for(j1=0; j1<ny1; j1++) { j0 = j1 * 2;
			for(i1=0; i1<nx1; i1++) { i0 = i1 * 2;
				ijk1 = i1 + nx1 * (j1 + ny1 * k1);
				ijk0 = i0 + nx0 * (j0 + ny0 * k0);

				if(i0+1 < nx0) { iadd =    1; } else { iadd = 0; }
				if(j0+1 < ny0) { jadd =  nx0; } else { jadd = 0; }
				if(k0+1 < nz0) { kadd = nxy0; } else { kadd = 0; }

				v0 = (int)fieldav[irep0][ijk0               ];
				v1 = (int)fieldav[irep0][ijk0+iadd          ];
				v2 = (int)fieldav[irep0][ijk0     +jadd     ];
				v3 = (int)fieldav[irep0][ijk0+iadd+jadd     ];
				v4 = (int)fieldav[irep0][ijk0          +kadd];
				v5 = (int)fieldav[irep0][ijk0+iadd     +kadd];
				v6 = (int)fieldav[irep0][ijk0     +jadd+kadd];
				v7 = (int)fieldav[irep0][ijk0+iadd+jadd+kadd];

				fieldav[irep1][ijk1] = (unsigned char)((float)0.125*(float)(v0+v1+v2+v3+v4+v5+v6+v7));
				vmax = v0;
				if(v1 > vmax) { vmax = v1; }
				if(v2 > vmax) { vmax = v2; }
				if(v3 > vmax) { vmax = v3; }
				if(v4 > vmax) { vmax = v4; }
				if(v5 > vmax) { vmax = v5; }
				if(v6 > vmax) { vmax = v6; }
				if(v7 > vmax) { vmax = v7; }
				fieldmax[irep1][ijk1] = (unsigned char)vmax;
				vmax = v0;
				if(v1 < vmin) { vmin = v1; }
				if(v2 < vmin) { vmin = v2; }
				if(v3 < vmin) { vmin = v3; }
				if(v4 < vmin) { vmin = v4; }
				if(v5 < vmin) { vmin = v5; }
				if(v6 < vmin) { vmin = v6; }
				if(v7 < vmin) { vmin = v7; }
				fieldmin[irep1][ijk1] = (unsigned char)vmin;
				if(vmin < 2) { fieldav[irep1][ijk1] = (unsigned char)v0; }
			}}}
//			printf("field averaged for irep = %2d\n", irep1);
		}
		irep1 *= 2;
	}


/*

	irep = 2;
	while(irep < 65) {
		if(iDoRep[irep] > 0) {
			irep0 = irep / 2;

			for(k=0; k<nz0; k += 2) { k2 = k/2;
			for(j=0; j<ny0; j += 2)   j2 = j/2; jkoff = nx0*(j + ny0*k); jkoff2 = nx2*(j2 + ny2*k2);  i2 = -1;
			for(i=0; i<nx0; i += 2) { i2++;

				v0 = (int)fieldav[irep0][jkoff+i         ];  v4 = (int)fieldav[irep0][jkoff+i+1         ];
				v1 = (int)fieldav[irep0][jkoff+i+nx0     ];  v5 = (int)fieldav[irep0][jkoff+i+1+nx0     ];
				v2 = (int)fieldav[irep0][jkoff+i    +nxy0];  v6 = (int)fieldav[irep0][jkoff+i+1    +nxy0];
				v3 = (int)fieldav[irep0][jkoff+i+nx0+nxy0];  v7 = (int)fieldav[irep0][jkoff+i+1+nx0+nxy0];

				fieldav[irep][jkoff2+i2] = (unsigned char)((float)0.125*(float)(v0+v1+v2+v3+v4+v5+v6+v7));
				vmax = v0;
				if(v1 > vmax) { vmax = v1; }
				if(v2 > vmax) { vmax = v2; }
				if(v3 > vmax) { vmax = v3; }
				if(v4 > vmax) { vmax = v4; }
				if(v5 > vmax) { vmax = v5; }
				if(v6 > vmax) { vmax = v6; }
				if(v7 > vmax) { vmax = v7; }
				fieldmax[irep][jkoff2+i2] = (unsigned char)vmax;
				vmax = v0;
				if(v1 < vmin) { vmin = v1; }
				if(v2 < vmin) { vmin = v2; }
				if(v3 < vmin) { vmin = v3; }
				if(v4 < vmin) { vmin = v4; }
				if(v5 < vmin) { vmin = v5; }
				if(v6 < vmin) { vmin = v6; }
				if(v7 < vmin) { vmin = v7; }
				fieldmin[irep][jkoff2+i2] = (unsigned char)vmin;
				if(vmin < 2) { fieldav[irep][jkoff2+i2] = (unsigned char)v0; }
			}}}
			printf("field averaged for irep = %2d\n", irep);
		}
		irep *= 2;
	}
*/

	// For every subregion currently in memory (pointed to by field)
	// for every tile in the Block tree (0 <= node < iNnodes)
	//    find the intersection
	//    if not empty: 1) copy intersecion into subregion of bob-tile and write to disk file
	//                  2) update max error tile
	//
	// Assumption: 1) Calling routine generate a sequence of regions which cover the entire domain.
	//             2) Region limits in each direction are multiples of the max.tile_rep.
	//             3) Region limits in XY span each tile that is updated

	// Used for generating offset in field array from ijk coords
	irn = curdim[0];
	jrn = curdim[1];
	krn = curdim[2];

	for(node=0; node<iNnodes; node++) {
		irep = tile_rep[node];

		// generate: Low and Hight inclusive limits on xyz (=ijk) ranges for "field" Region
		irl = curoff[0];   irh = curoff[0] + curdim[0] - 1;
		jrl = curoff[1];   jrh = curoff[1] + curdim[1] - 1;
		krl = curoff[2];   krh = curoff[2] + curdim[2] - 1;

		// save actual IJK limits of array field
		irl0 = irl;                        irh0 = irh;
		jrl0 = jrl;                        jrh0 = jrh;
		krl0 = krl;                        krh0 = krh;

		// If Region is at domain edge, then -- conceptually -- it includes a boundary of depth irep.
		// Values in this boundary region are gotten from the closest points interior to array "field"
		if(irl == 0) { irl -= irep; }      if(irh == nx-1) { irh += 2 * irep; }
		if(jrl == 0) { jrl -= irep; }      if(jrh == ny-1) { jrh += 2 * irep; }
		if(krl == 0) { krl -= irep; }      if(krh == nz-1) { krh += 2 * irep; }

		// dimensions of tile
		itn = tile_dim[node][0];
		jtn = tile_dim[node][1];
		ktn = tile_dim[node][2];

		// generate: Low and Hight inclusive limits on xyz (=IJK) ranges for block tree Tile
		itl = irep*tile_offset[node][0];   ith = irep*(tile_offset[node][0] + tile_dim[node][0]) - 1;
		jtl = irep*tile_offset[node][1];   jth = irep*(tile_offset[node][1] + tile_dim[node][1]) - 1;
		ktl = irep*tile_offset[node][2];   kth = irep*(tile_offset[node][2] + tile_dim[node][2]) - 1;

		// generate: Low and Hight inclusive limits on z (=K) range for Intersection of Tile and Region
		if(krl > ktl) { kil = krl; } else { kil = ktl; }
		if(krh < kth) { kih = krh; } else { kih = kth; }

		// if the Low limit of the intersection > the High limit then there is no intersection
		iIntersection = 1;
		if(kil > kih) { iIntersection = 0; }

		// if any part of the tile in XY (=IJ) directions is not contained in the array field Region
		// then dont do tile.  Assume this block-tree tile (=node) is covered in some other call to this subroutine.
		if(itl < irl  ||  ith > irh) { iIntersection = 0; }
		if(jtl < jrl  ||  jth > jrh) { iIntersection = 0; }

		if(iIntersection == 1) {

			// Generate IJK Low and High coords in tile Units
			itlu = tile_offset[node][0];   ithu = tile_offset[node][0] + tile_dim[node][0];
			jtlu = tile_offset[node][1];   jthu = tile_offset[node][1] + tile_dim[node][1];
			ktlu = tile_offset[node][2];   kthu = tile_offset[node][2] + tile_dim[node][2];

			// This if-block is where all the copy to tiles and output is done.

			// 1) Array fField is copied, or averaged if irep>1, as floats into array fSlab
			//    max and min fField values (for each zone) are gathered into fSlabMax, fSlabMin arrays
			//    The three fSlab arrays are always in the tile space.
			kilu =  kil    / irep;
			kihu = (kih+1) / irep;
			nslab = itn * jtn * (kihu - kilu);		// size of slab
//			nx0 = fulldim[0] / irep;
//			ny0 = fulldim[1] / irep;
//			nz0 = fulldim[2] / irep;

			nx0 = (curdim[0] + irep - 1) / irep;
			ny0 = (curdim[1] + irep - 1) / irep;
			nz0 = (curdim[2] + irep - 1) / irep;

			ijkd = 0;
			if(irep == 1) {
				for(k=kilu; k<kihu; k++) {
					ks = k - curoff[2] / irep;
					if(ks <     0) { ks =     0; }
					if(ks > nz0-1) { ks = nz0-1; }
//					ks = ks - curoff[2] / irep;
					for(j=jtlu; j<jthu; j++) {
						js = j;
						if(js <     0) { js =     0; }
						if(js > ny0-1) { js = ny0-1; }
						jkoff = nx0*(js + ny0*ks);

/*********************************************************************************
						itlu0 = itlu;   if(itlu <     0) { itlu0 =     0; }
						ithu0 = ithu;   if(ithu > nx0-1) { ithu0 = nx0-1; }

						if(itlu < 0) {
							ucSlab[ijkd] = fieldav[irep][itlu0+jkoff];
							ijkd++;
						}

						memcpy(&ucSlab[ijkd], &fieldav[irep][itlu0+jkoff], ithu0-itlu0);
						ijkd += ithu0-itlu0;

						if(ithu > nx0-1) {
							ucSlab[ijkd] = fieldav[irep][ithu0+jkoff];
							ijkd++;
						}
 *********************************************************************************/

/*********************************************************************************/
						for(i=itlu; i<ithu; i++) {
							is = i;
							if(is <     0) { is =     0; }
							if(is > nx0-1) { is = nx0-1; }
							ijks = is + jkoff;
							ucSlab[ijkd] = fieldav[irep][ijks];
							ijkd++;
							ijks++;
						}
/*********************************************************************************/

/*********************************************************************************
						itlu0 = itlu;
						ithu0 = ithu;
						if(itlu < 0) {
							itlu0 =   0;
							ucSlab[ijkd] = fieldav[irep][jkoff];
							ijkd++;
						}

						ijks = itlu0+jkoff;

						if(ithu > nx0-1) {
							ithu0 = nx0;
							memcpy(&ucSlab[ijkd], &fieldav[irep][ijks], ithu0-itlu0);
							ijkd += ithu0-itlu0;

							ucSlab[ijkd] = fieldav[irep][jkoff+nx0-1];
							ijkd++;
						} else {
							memcpy(&ucSlab[ijkd], &fieldav[irep][ijks], ithu0-itlu0);
							ijkd += ithu0-itlu0;
						}
 *********************************************************************************/

					}
				}
			} else {
				for(k=kilu; k<kihu; k++) {
					ks = k - curoff[2] / irep;
					if(ks <     0) { ks =     0; }
					if(ks > nz0-1) { ks = nz0-1; }
//					ks = ks - curoff[2] / irep;
					for(j=jtlu; j<jthu; j++) {
						js = j;
						if(js <     0) { js =     0; }
						if(js > ny0-1) { js = ny0-1; }
						jkoff = nx0*(js + ny0*ks);

						for(i=itlu; i<ithu; i++) {
							is = i;
							if(is <     0) { is =     0; }
							if(is > nx0-1) { is = nx0-1; }
							ijks = is + jkoff;
							ucSlab[ijkd] = fieldav[irep][ijks];
							fSlabMin[ijkd] = fieldmin[irep][ijks];
							fSlabMax[ijkd] = fieldmax[irep][ijks];
							ijkd++;
						}
					}
				}
			}

			// 3) Max error for node is updated based on min and max values under each voxel
			//    Only do this for tiles of reduced resolution, otherwise error is always 0
			if(irep > 1) {
				maxerrtmp = tile_error[node];
				for(i=0; i<nslab; i++) {
					iErr = (int)ucSlab[i] - (int)fSlabMin[i];
					if(maxerrtmp < iErr) { maxerrtmp = iErr; }

					iErr = (int)fSlabMax[i] - (int)ucSlab[i];
					if(maxerrtmp < iErr) { maxerrtmp = iErr; }
				}
				tile_error[node] = maxerrtmp;
			}
			// 4) ucSlab data is written to disk file
			//    Assume slab spans X- and Y-dims of tile
			i64Offset = tile_byteoff[node] + (off64_t)(itn * jtn * (kil-ktl)/irep);

//			printf("XYS-Dim = %3d %3d %2d  nbytes=%2d  val=%3d  seek=%15lld\n",
//				itn, jtn, (kih-kil+1)/irep, nslab, (int)ucSlab[nslab/3], i64Offset);
//			printf("XYS-Dim = %3d %3d %2d  irep=%2d  ki[lh]=%3d %3d  kt[lh]=%3d %3d\n",
//				itn, jtn, (kih-kil+1)/irep, irep, kil, kih, ktl, kth);

			lseek64(fd, i64Offset, SEEK_SET);
			write(fd, ucSlab, nslab);

		}							// end of iIntersection if-block

	}								// end of node loop

	// Output Tree
//	printf("z:offset=%d   z:size=%d   z:offset+size=%d\n", curoff[2],curdim[2],curoff[2]+curdim[2]);
	if(curoff[2]+curdim[2] >= nz) {
//		printf("nx=%d   ny=%d  nz=%d   Block=%d\n", nx, ny, nz, iBlock);

		iOff = 0;
		for(node=0; node<iNnodes; node++) {

			sprintf(cLine, "%s %10ld %2d   %4d %4d %4d   %5d %5d %5d %8d %3d\n",
				pfile,			  (long)tile_byteoff[node],		tile_rep[node],
				tile_dim[node][0],		tile_dim[node][1],		tile_dim[node][2],
				tile_offset[node][0],	tile_offset[node][1],	tile_offset[node][2],
				tile_parent[node],		tile_error[node]							);

			nChar = 0;
			while ( (int)cLine[nChar] != (int)'\n'  &&  nChar < 256 ) { nChar++; }
			nChar++;			
//			if(nChar > 120) { printf("nChar = %d\n", nChar); nChar = 120; }
			for(i=0; i<nChar; i++) { ucSlab[iOff+i] = (unsigned char)cLine[i]; }
			iOff += nChar;
		}
		
		if(fLimits[0] < fLimits[1]) {
			sprintf(cLine, "XYZ_Limits %f %f %f %f %f %f\n",
				fLimits[0], fLimits[1], fLimits[2], fLimits[3], fLimits[4], fLimits[5]);

			nChar = 0;
			while ( (int)cLine[nChar] != (int)'\n'  &&  nChar < 256 ) { nChar++; }
			nChar++;
//			if(nChar > 200) { printf("nChar = %d\n", nChar); nChar = 120; }
			for(i=0; i<nChar; i++) { ucSlab[iOff+i] = (unsigned char)cLine[i]; }
			iOff += nChar;
		}

		// Pad header with a whole bunch of charage returns
		sprintf(cLine, "\n");
		for(i=0; i<256; i++) { ucSlab[iOff+i] = (unsigned char)cLine[0]; }
		nChar += iOff + 256;

		lseek64(fd, 0, SEEK_SET);
		write(fd, ucSlab, nChar);
		close(fd);
	}
}

void TreeScanOffsets(int node, off64_t *pbyteoff, off64_t tile_byteoff[MAX_NODES],
					 int irep, int tile_rep[MAX_NODES], int tile_dim[MAX_NODES][3],
					 int tile_N_children[MAX_NODES], int tile_children[MAX_NODES][MAX_CHILDREN])
{
	if(irep == tile_rep[node]) {
		tile_byteoff[node] = *pbyteoff;
		*pbyteoff += (off64_t)(FILE_BLOCK*(1 + (tile_dim[node][0]*tile_dim[node][1]*tile_dim[node][2]-1)/
											   (FILE_BLOCK)));
	} else {
		int ichild;
		for(ichild=0; ichild<tile_N_children[node]; ichild++) {
			TreeScanOffsets(tile_children[node][ichild], pbyteoff, tile_byteoff,
							irep, tile_rep, tile_dim,
							tile_N_children, tile_children                       );
		}
	}

	return;
}

















